<?php declare( strict_types=1 );

namespace Simplyin\Simplyin_W_Plugin\Lib\Base_Plugin;

use Exception;
use Simplyin\Simplyin_W_Plugin\Lib\Base_Plugin\Helper\Core;
use Simplyin\Simplyin_W_Plugin\Lib\Base_Plugin\Logger\Logger_Interface;
use Simplyin\Simplyin_W_Plugin\Lib\Base_Plugin\Logger\Null_Logger;
use function function_exists;

abstract class Abstract_Base_Plugin {

	use Tools, Environment;

	private static $config;

	/**
	 * @var self
	 */
	public static $initial_instance;

	/**
	 * @var Logger_Interface
	 */
	private static $logger;

	/**
	 * @param array $config
	 *
	 * @return void
	 * @throws Exception
	 */
	public function execute( array $config ) {
		self::$config = $config;
		if ( ! self::$initial_instance ) {
			self::$initial_instance = $this;
		}

		register_activation_hook(
			$this->get__file__(),
			[ $this, 'plugin_activate_actions' ]
		);
		register_deactivation_hook(
			$this->get__file__(),
			[ $this, 'plugin_deactivate_actions' ]
		);

		$this->init_translations();
		$this->before_init();

		if ( ! self::$logger ) {
			self::$logger = new Null_Logger();
		}

		add_action( 'init', function () {
			$this->enqueue_scripts();
			$this->init();
		} );
		add_action( 'plugins_loaded', function () use ( $config ) {

			$this->plugins_loaded_hooks();
		} );
	}

	public function get_features_config(): ?Features_Config_Interface {
		return null;
	}

	public function plugin_activate_actions() {

	}

	public function plugin_deactivate_actions() {

	}

	/**
	 * @return void
	 * @throws Exception
	 */
	private function init_translations() {

	}

	abstract protected function before_init();

	abstract protected function init();

	private function enqueue_scripts() {
		add_action( 'admin_enqueue_scripts',
			[ $this, 'enqueue_dashboard_scripts' ] );
		add_action( 'wp_enqueue_scripts',
			[ $this, 'enqueue_frontend_scripts' ] );
	}

	abstract protected function plugins_loaded_hooks();

	/**
	 * @return Alerts
	 */
	public function alerts(): Alerts {
		return new Alerts();
	}

	public function add_slug_prefix( string $text ): string {
		return $this->get_from_config( 'slug' ) . '_' . $text;
	}

	/**
	 * @return Logger_Interface
	 */
	public function get_logger(): Logger_Interface {
		return self::$logger;
	}

	/**
	 * @throws Exception
	 */
	public function get_woocommerce_logger(): Logger_Interface {
		return new Woocommerce_Logger( $this->get_from_config( 'slug' ) );
	}

	/**
	 * @throws Exception
	 */
	public function locate_template(
		string $template,
		array $args = [],
		?string $template_dir = null
	) {
		$directory_separator    = DIRECTORY_SEPARATOR;
		$template_absolute_path = $this->get_plugin_templates_dir($template_dir) . $directory_separator . $template;

		if ( '' === locate_template( [ $template_absolute_path ],
				true,
				$args ) ) {
			extract( $args );
			include $template_absolute_path;
		}
	}

	abstract public function enqueue_frontend_scripts();

	abstract public function enqueue_dashboard_scripts();
}
