<?php
/**
 * Copyright 2024-2025 Simply.IN Sp. z o.o.
 *
 * NOTICE OF LICENSE
 *
 * Licensed under the EUPL-1.2 or later.
 * You may not use this work except in compliance with the Licence.
 *
 * Copy of the Licence is available at:
 * https://joinup.ec.europa.eu/software/page/eupl
 * It is bundled with this package in the file LICENSE.txt
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the Licence is distributed on an as is basis,
 * without warranties or conditions of any kind, either express or implied.
 * Check the Licence for the specific language governing permissions
 * and limitations under the License.
 *
 * @author   Simply.IN Sp. z o.o.
 * @copyright 2024-2025 Simply.IN Sp. z o.o.
 * @license   https://joinup.ec.europa.eu/software/page/eupl
 */

use PrestaShop\PrestaShop\Core\Payment\PaymentOption;

require_once dirname(__FILE__) . '/models/SimplyInPoint.php';
require_once dirname(__FILE__) . '/models/SimplyInOrder.php';
require_once dirname(__FILE__) . '/models/SimplyInOrderSync.php';
require_once dirname(__FILE__) . '/models/SimplyInEmailCache.php';
require_once dirname(__FILE__) . '/classes/SimplyInApi.php';
require_once dirname(__FILE__) . '/classes/SimplyInSession.php';
require_once dirname(__FILE__) . '/models/SimplyInPayments.php';

if (!defined('_PS_VERSION_')) {
    exit;
}
class Simplyin extends PaymentModule
{
    /** @var SimplyInSession|null This is an instance of SimplyInSession */
    public $simplyin_session = null;
    /** @var SimplyInApi This is an instance of SimplyInApi */
    public $api = false;
    public $backend_url = "https://prod.backend.simply.in/api/";
    public $merchand_api_key = '';

    private static $simply_update_settings_called = false;

    public function __construct()
    {
        $this->name = 'simplyin';
        $this->tab = 'shipping_logistics';
        $this->version = '2.0.9';
        $this->author = 'Simply.IN';
        $this->need_instance = 1;
        $this->bootstrap = true;
        parent::__construct();
        $this->controllers = ['payment', 'validation'];

        $this->displayName = $this->l('Simply.IN');
        $this->description = "Simply.IN Super easy and fast 1-click shopping  ";
        $this->ps_versions_compliancy = ['min' => '1.7', 'max' => '8.99.99'];
        if ($this->api === false) {
            $this->loadObjects();
        }
    }

    public function install()
    {
        return parent::install()
            && $this->installDefaultConfiguration()
            && $this->registerHooks()
            && $this->installSql()
            && $this->installOrderStates();
    }

    public function registerHooks()
    {
        return $this->registerHook('actionAdminOrdersTrackingNumberUpdate')
            && $this->registerHook('actionCustomerLogoutBefore')
            && $this->registerHook('actionOrderStatusPostUpdate')
            && $this->registerHook('actionCustomerAccountUpdate')
            && $this->registerHook('displayHeader')
            && $this->registerHook('displayBackOfficeHeader')
            && $this->registerHook('displayAdminOrderMain')
            && $this->registerHook('actionCarrierUpdate')
            && $this->registerHook('actionValidateOrder')
            && $this->registerHook('paymentOptions')
            && $this->registerHook('displayOrderConfirmation')
        ;

    }

    public function installSql()
    {
        return SimplyInOrder::installSql() && SimplyInPoint::installSql() && SimplyInOrderSync::installSql() && SimplyInEmailCache::installSql() && SimplyInPayments::installSql();
    }

    public function installDefaultConfiguration()
    {
        Configuration::updateValue('SIMPLYIN_TRANS_HISTORY_LENGTH_DAY', '180');
        Configuration::updateValue('SIMPLYIN_SAVE_CHECKBOX', false);
        Configuration::updateValue('SIMPLYIN_SWITCH_LASTCHANGE', '');
        Configuration::updateValue('SIMPLYIN_LOGISTICSPARTNERSORDER', '');

        return true;
    }

    public function uninstall()
    {
        SimplyInOrder::uninstallSql();
        SimplyInPoint::uninstallSql();
        SimplyInOrderSync::uninstallSql();
        SimplyInEmailCache::uninstallSql();
        SimplyInPayments::uninstallSql();
        $this->unregisterHooks();

        return parent::uninstall();
    }

    public function unregisterHooks()
    {
        $this->unregisterHook('actionAdminOrdersTrackingNumberUpdate');
        $this->unregisterHook('actionCustomerLogoutBefore');
        $this->unregisterHook('actionOrderStatusPostUpdate');
        $this->unregisterHook('actionCustomerAccountUpdate');
        $this->unregisterHook('displayHeader');
        $this->unregisterHook('displayBackOfficeHeader');
        $this->unregisterHook('displayAdminOrderMain');
        $this->unregisterHook('actionCarrierUpdate');
        $this->unregisterHook('actionValidateOrder');
        $this->unregisterHook('paymentOptions');
        $this->unregisterHook('displayOrderConfirmation');

        return true;
    }

    public function getContent()
    {
        if (Tools::isSubmit('submitSimplyinModule')) {
            $this->postProcess();
            SimplyInOrderSync::refreshOrdersSync();
            SimplyInOrderSync::synchroneOrders();
        }
        $this->context->smarty->assign([
            'cron_link' => $this->context->link->getModuleLink($this->name, 'cron', ['secure_key' => $this->getCronSecureKey()], true),
            'module_path' => $this->_path,
        ]);
        $output = $this->context->smarty->fetch($this->local_path . 'views/templates/admin/help.tpl');

        return $output . $this->renderForm();
    }

    protected function renderForm()
    {
        $helper = new HelperForm();

        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->module = $this;
        $helper->default_form_language = $this->context->language->id;
        $helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG', 0);

        $helper->identifier = $this->identifier;
        $helper->submit_action = 'submitSimplyinModule';
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false)
            . '&configure=' . $this->name . '&tab_module=' . $this->tab . '&module_name=' . $this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');

        $helper->tpl_vars = [
            'fields_value' => $this->getConfigFormValues(),
            'languages' => $this->context->controller->getLanguages(),
            'id_language' => $this->context->language->id,
        ];

        return $helper->generateForm($this->getConfigForm());
    }

    protected function getConfigForm()
    {
        $this->context->smarty->assign('localPath', _PS_BASE_URL_ . $this->_path);
        $forms = [];
        $forms[] = [
            'form' => [
                'legend' => [
                    'title' => $this->l('Settings'),
                    'icon' => 'icon-cogs',
                ],
                'input' => [
                    [
                        'type' => 'text',
                        'label' => $this->l('Secret key'),
                        'name' => 'SIMPLYIN_SECRET_KEY',
                        'suffix' => '<span class="simplyin-toggle">👁️</span>',
                    ],
                    [
                        'type' => 'switch',
                        'id' => 'simply-switch',
                        'label' => $this->l('Create SimplyIn account checkbox is checked by default'),
                        'name' => 'SIMPLYIN_SAVE_CHECKBOX',
                        'is_bool' => true,
                        'values' => [
                            [
                                'id' => 'active_on',
                                'value' => 1,
                                'label' => $this->l('Yes'),
                            ],
                            [
                                'id' => 'active_off',
                                'value' => 0,
                                'label' => $this->l('No'),
                            ],
                        ],
                    ],
                ],
                'submit' => [
                    'title' => $this->l('Save'),
                ],
            ],
        ];
        $carriers = $this->getAllAvailableShippingMethods();

        array_unshift($carriers, ['id_carrier' => 0, 'name' => '--']);

        $forms[] = [
            'form' => [
                'legend' => [
                    'title' => $this->l('Carrier settings'),
                    'icon' => 'icon-cogs',
                ],
                'input' => [
                    [
                        'type' => 'select',
                        'name' => 'SIMPLYIN_DPD_CARRIER_ID',
                        'label' => $this->l('DPD parcel locker carrier'),
                        'options' => [
                            'query' => $carriers,
                            'id' => 'id_carrier',
                            'name' => 'name',
                        ],
                    ],
                    [
                        'type' => 'select',
                        'name' => 'SIMPLYIN_DHL_CARRIER_ID',
                        'label' => $this->l('DHL parcel locker carrier'),
                        'options' => [
                            'query' => $carriers,
                            'id' => 'id_carrier',
                            'name' => 'name',
                        ],
                    ],
                    [
                        'type' => 'select',
                        'name' => 'SIMPLYIN_FEDEX_CARRIER_ID',
                        'label' => $this->l('FedEx parcel locker carrier'),
                        'options' => [
                            'query' => $carriers,
                            'id' => 'id_carrier',
                            'name' => 'name',
                        ],
                    ],
                    [
                        'type' => 'select',
                        'name' => 'SIMPLYIN_INPOST_CARRIER_ID',
                        'label' => $this->l('InPost parcel locker carrier'),
                        'options' => [
                            'query' => $carriers,
                            'id' => 'id_carrier',
                            'name' => 'name',
                        ],
                    ],
                    [
                        'type' => 'select',
                        'name' => 'SIMPLYIN_ORLEN_CARRIER_ID',
                        'label' => $this->l('Orlen parcel locker carrier'),
                        'options' => [
                            'query' => $carriers,
                            'id' => 'id_carrier',
                            'name' => 'name',
                        ],
                    ],
                    [
                        'type' => 'select',
                        'name' => 'SIMPLYIN_UPS_CARRIER_ID',
                        'label' => $this->l('UPS parcel locker carrier'),
                        'options' => [
                            'query' => $carriers,
                            'id' => 'id_carrier',
                            'name' => 'name',
                        ],
                    ],
                    [
                        'type' => 'select',
                        'name' => 'SIMPLYIN_DEFAULT_CARRIER_ID',
                        'label' => $this->l('Default carrier'),
                        'options' => [
                            'query' => $carriers,
                            'id' => 'id_carrier',
                            'name' => 'name',
                        ],
                    ],
                ],
                'submit' => [
                    'title' => $this->l('Save'),
                ],
            ],
        ];

        $order_states = OrderState::getOrderStates($this->context->language->id);
        $forms[] = [
            'form' => [
                'legend' => [
                    'title' => $this->l('Payment settings'),
                    'icon' => 'icon-cogs',
                ],
                'input' => [
                    [
                        'type' => 'switch',
                        'name' => 'SIMPLYIN_PAYMENTS',
                        'label' => $this->l('Enable SimplyIN payments'),
                        'is_bool' => true,
                        'values' => [
                            [
                                'id' => 'active_on',
                                'value' => 1,
                                'label' => $this->l('Yes'),
                            ],
                            [
                                'id' => 'active_off',
                                'value' => 0,
                                'label' => $this->l('No'),
                            ],
                        ],
                    ],
                    [
                        'type' => 'select',
                        'name' => 'SIMPLYIN_PAYMENT_STATUS_PENDING',
                        'label' => $this->l('Order status for pending payments'),
                        'options' => [
                            'query' => $order_states,
                            'id' => 'id_order_state',
                            'name' => 'name',
                        ],
                    ],
                    [
                        'type' => 'select',
                        'name' => 'SIMPLYIN_PAYMENT_STATUS_CANCELED',
                        'label' => $this->l('Order status for canceled payments'),
                        'options' => [
                            'query' => $order_states,
                            'id' => 'id_order_state',
                            'name' => 'name',
                        ],
                    ],
                ],
                'submit' => [
                    'title' => $this->l('Save'),
                ],
            ],
        ];
        return $forms;
    }

    protected function getConfigFormValues()
    {
        return [
            'SIMPLYIN_SECRET_KEY' => Configuration::get('SIMPLYIN_SECRET_KEY'),
            'SIMPLYIN_SAVE_CHECKBOX' => Configuration::get('SIMPLYIN_SAVE_CHECKBOX'),
            'SIMPLYIN_DPD_CARRIER_ID' => Configuration::get('SIMPLYIN_DPD_CARRIER_ID'),
            'SIMPLYIN_DHL_CARRIER_ID' => Configuration::get('SIMPLYIN_DHL_CARRIER_ID'),
            'SIMPLYIN_FEDEX_CARRIER_ID' => Configuration::get('SIMPLYIN_FEDEX_CARRIER_ID'),
            'SIMPLYIN_INPOST_CARRIER_ID' => Configuration::get('SIMPLYIN_INPOST_CARRIER_ID'),
            'SIMPLYIN_ORLEN_CARRIER_ID' => Configuration::get('SIMPLYIN_ORLEN_CARRIER_ID'),
            'SIMPLYIN_UPS_CARRIER_ID' => Configuration::get('SIMPLYIN_UPS_CARRIER_ID'),
            'SIMPLYIN_DEFAULT_CARRIER_ID' => Configuration::get('SIMPLYIN_DEFAULT_CARRIER_ID'),
            'SIMPLYIN_PAYMENT_STATUS_PENDING' => Configuration::get('SIMPLYIN_PAYMENT_STATUS_PENDING'),
            'SIMPLYIN_PAYMENT_STATUS_CANCELED' => Configuration::get('SIMPLYIN_PAYMENT_STATUS_CANCELED'),
            'SIMPLYIN_PAYMENTS' => Configuration::get('SIMPLYIN_PAYMENTS')
        ];
    }

    protected function postProcess()
    {
        $form_values = $this->getConfigFormValues();
        foreach (array_keys($form_values) as $key) {
            $value = Tools::getValue($key);
            Configuration::updateValue($key, $value);
        }
        if ($this->simplyInUpdateMerchantSettings(Tools::getValue('SIMPLYIN_SECRET_KEY'), true)) {
            $this->context->controller->confirmations[] = $this->l('Successfully connected to SimplyIN.');
            Configuration::updateValue('SIMPLYIN_CONNECTED', true);
        } else {
            $this->context->controller->errors[] = $this->l('Error: Cannot connect to SimplyIN please check your API key');
            Configuration::updateValue('SIMPLYIN_CONNECTED', false);
        }
        $this->context->controller->confirmations[] = $this->l('Configuration updated.');
    }

    public function hookActionOrderStatusPostUpdate($params)
    {
        if (Configuration::get('SIMPLYIN_CONNECTED')) {
            if (isset($params['id_order'])) {
                $status = $params['newOrderStatus']->template;
                $id_order = $params['id_order'];

                $object = SimplyInOrder::getByIdOrder($id_order);
                if ($object) {
                    $allowed = [
                        'shipped',
                        'payment',
                    ];
                    if (!in_array($status, $allowed, true)) {
                        return;
                    }
                    $this->api->simplyInChangeOrderState($id_order, $status);
                }
            }
        }
    }

    public function hookDisplayAdminOrderMain($params)
    {
        $docs = Configuration::get('SIMPLYIN_DOCS_INFO');
        if (isset($params['id_order'])) {
            $id_order = $params['id_order'];
            $order = new Order($id_order);
            $id_cart = $order->id_cart;
            if ($id_cart) {
                $selected_point = SimplyInPoint::getByIdCart($id_cart);
                if ($selected_point->id) {
                    $this->context->smarty->assign([
                        'selected_point' => $selected_point,
                        'module_path' => $this->_path,
                    ]);
                    return $this->display(__FILE__, 'views/templates/admin/order.tpl');
                }
            }
        }
    }

    public function hookDisplayBackOfficeHeader()
    {
        if (Tools::getValue('configure') == $this->name) {
            $this->context->controller->addJquery();
            $this->context->controller->addJqueryPlugin('select2');
            $this->context->controller->addJS($this->_path . 'views/js/simplyin_back.js');
            $this->context->controller->addCSS($this->_path . 'views/css/simplyin_back.css');
        }
    }

    public function checkController()
    {
        $controller = Tools::getValue('controller');
        if ($controller == 'order' || $controller == 'cart' || $controller == 'orderconfirmation' || $controller == 'authentication') {
            return true;
        }
        return false;
    }

    public function getOPC()
    {
        static $opc = null;
        if ($opc !== null) {
            return $opc;
        }

        $thecheckout = Module::getInstanceByName('thecheckout');
        if ($thecheckout && $thecheckout->active) {
            $opc = 'thecheckout';
        } else {
            $opc = 'classic';
        }

        return $opc;
    }

    public function hookDisplayHeader()
    {
        if (!Configuration::get('SIMPLYIN_CONNECTED')) {
            return;
        }

        if (!$this->checkController()) {
            return;
        }

        Media::addJsDef($this->getJsDefaultHeader());
        $this->context->controller->addJquery();

        $base = 'modules/'.$this->name.'/';

        $this->context->controller->registerJavascript(
            'module-'.$this->name.'-front',
            $base.'views/js/front.js',
            ['position' => 'bottom', 'priority' => 150, 'server' => 'local', 'version' => $this->version]
        );

        if ($this->getOPC() == 'classic') {
            $this->context->controller->registerJavascript(
                'module-'.$this->name.'-front-payment',
                $base.'views/js/front_payment.js',
                ['position' => 'bottom', 'priority' => 151, 'server' => 'local', 'version' => $this->version]
            );
            $this->context->controller->registerStylesheet(
                'module-'.$this->name.'-front-payment-css',
                $base.'views/css/simplyin_front_payment.css',
                ['media' => 'all', 'priority' => 150, 'server' => 'local', 'version' => $this->version]
            );
        } elseif ($this->getOPC() == 'thecheckout') {
            $this->context->controller->registerJavascript(
                'module-'.$this->name.'-thecheckout',
                $base.'views/js/thecheckout.js',
                ['position' => 'bottom', 'priority' => 151, 'server' => 'local', 'version' => $this->version]
            );
            if (Configuration::get('SIMPLYIN_PAYMENTS')) {
                $this->context->controller->registerJavascript(
                    'module-'.$this->name.'-front-payment',
                    $base.'views/js/front_payment.js',
                    ['position' => 'bottom', 'priority' => 152, 'server' => 'local', 'version' => $this->version]
                );
                $this->context->controller->registerJavascript(
                    'module-'.$this->name.'-front-payment-thecheckout',
                    $base.'views/js/front_payment_thecheckout.js',
                    ['position' => 'bottom', 'priority' => 153, 'server' => 'local', 'version' => $this->version]
                );
                $this->context->controller->registerStylesheet(
                    'module-'.$this->name.'-front-payment-css',
                    $base.'views/css/simplyin_front_payment.css',
                    ['media' => 'all', 'priority' => 150, 'server' => 'local', 'version' => $this->version]
                );
            }
        }

        $this->context->controller->registerStylesheet(
            'module-'.$this->name.'-front',
            $base.'views/css/simplyin_front.css',
            ['media' => 'all', 'priority' => 150, 'server' => 'local', 'version' => $this->version]
        );
    }

    public function getAllAvailableShippingMethods()
    {
        $carriers = Carrier::getCarriers($this->context->language->id, true, false, false, null, Carrier::ALL_CARRIERS);
        return $carriers;
    }

    public function getAsyncToken()
    {
        return md5(_COOKIE_KEY_ . _COOKIE_IV_ . $this->name);
    }

    public function getAvailableCarriers()
    {
        $carriers = [];
        $config = $this->getConfigFormValues();
        if ($config['SIMPLYIN_DPD_CARRIER_ID']) {
            $carriers[] = 'dpd';
        }
        if ($config['SIMPLYIN_UPS_CARRIER_ID']) {
            $carriers[] = 'ups';
        }
        if ($config['SIMPLYIN_DHL_CARRIER_ID']) {
            $carriers[] = 'dhl';
        }
        if ($config['SIMPLYIN_FEDEX_CARRIER_ID']) {
            $carriers[] = 'fedex';
        }
        if ($config['SIMPLYIN_INPOST_CARRIER_ID']) {
            $carriers[] = 'inpost';
        }
        if ($config['SIMPLYIN_ORLEN_CARRIER_ID']) {
            $carriers[] = 'orlen';
        }
        return $carriers;
    }

    public function getCarriersKey()
    {
        return [
            'SIMPLYIN_DPD_CARRIER_ID',
            'SIMPLYIN_UPS_CARRIER_ID',
            'SIMPLYIN_DHL_CARRIER_ID',
            'SIMPLYIN_FEDEX_CARRIER_ID',
            'SIMPLYIN_INPOST_CARRIER_ID',
            'SIMPLYIN_ORLEN_CARRIER_ID',
            'SIMPLYIN_DEFAULT_CARRIER_ID',
        ];
    }

    public function getJsDefaultHeader($override = [])
    {
        $context = $this->context;
        $currentLanguage = $context->language;
        $customer = $context->customer;
        $hide_phone_field = false;
        $new_account_phone = false;
        if ($customer && !empty($customer->email)) {
            $hide_phone_field = $this->checkUserExist($customer->email);
            $new_account_phone = $this->getNewAccountPhone();
        }

        if ($this->context->cart->id) {
            $point = SimplyInPoint::getByIdCart($this->context->cart->id);
        } else {
            $point = new SimplyInPoint();
        }

        $id_address_delivery = $this->simplyin_session->get('simplyin_id_address_delivery', false);
        $api_logged = trim($this->simplyin_session->get('simplyin_auth_token') != '') ? true : false;
        $id_carrier = isset($this->context->cart->id_carrier) ? (int) $this->context->cart->id_carrier : 0;
        $checkout = $this->getCheckout();
        $return = [
            'simplyin' => [
                'checkout' => $checkout,
                'ajax_token' => $this->getAjaxToken(),
                'cart_token' => $this->getCartToken(),
                'files_url' => $this->_path,
                'id_carrier' => $id_carrier,
                'controllers' => [
                    'ajax' => $this->context->link->getModuleLink('simplyin', 'ajax'),
                    'payment_ajax' => $this->context->link->getModuleLink('simplyin', 'paymentajax'),
                ],
                'customer_email' => $this->getCustomerEmail(),
                'customer' => [
                    'has_selected_address' => (bool) $id_address_delivery,
                    'is_guest' => $customer->isGuest(),
                    'logged' => $customer->isLogged(),
                    'has_token' => $api_logged,
                    'email' => $customer->email,
                    'point' => [
                        'label' => $point->label,
                        'locker_id' => $point->lockerId,
                    ],
                ],
                'currentLanguage' => $currentLanguage,
                'labels' => [
                    'By paying, I accept the terms and [conditions] and [privacy policy] of %s' => $this->l('By paying, I accept the terms and [conditions] and [privacy policy] of %s'),
                    'Please select a payment method' => $this->l('Please select a payment method'),
                    'BLIK code rejected' => $this->l('BLIK code rejected'),
                    'cancel' => $this->l('Cancel'),
                    'change' => $this->l('Change'),
                    'code_invalid' => $this->l('Invalid code'),
                    'confirm_delete' => $this->l('Confirm delete address?'),
                    'data_completed' => $this->l('Data completed'),
                    'delete' => $this->l('Delete'),
                    'delete_header' => $this->l('Delete address'),
                    'edit' => $this->l('Edit'),
                    'Invalid BLIK code' => $this->l('Invalid BLIK code'),
                    'Invalid tokenization response' => $this->l('Invalid tokenization response'),
                    'Problem processing card' => $this->l('Problem processing card'),
                    'Problem with BLIK processing' => $this->l('Problem with BLIK processing'),
                ],

                'hide_phone_field' => $hide_phone_field,
                'new_account_phone' => $new_account_phone,
                'save_checkbox' => Configuration::get('SIMPLYIN_SAVE_CHECKBOX'),
                'select_customer_privacy' => false,
                'select_psgdpr' => false,
                'submit_guest_form' => false,
            ],
        ];
        if (Tools::getValue('controller') == 'orderconfirmation') {
            $ajax_send = true;
            $id_cart = (int) Tools::getValue('id_cart');
            $id_module = (int) Tools::getValue('id_module');
            $ajax_token = $this->getAjaxToken();
            $cart_token = $this->getCartToken($id_cart);
            $payment_check_token = $this->getPaymentCheckToken($ajax_token, $cart_token);
            if ($id_module) {
                $module = Module::getInstanceById($id_module);
                if ($module->name == 'simplyin') {
                    $ajax_send = false;
                    $return['simplyin'] = [
                        'order_sended' => true,
                        'controllers' => [
                            'payment_ajax' => $this->context->link->getModuleLink('simplyin', 'paymentajax'),
                        ],
                        'files_url' => $this->_path,
                        'id_cart' => $id_cart,
                        'id_module' => $id_module,
                        'key' => Tools::getValue('key'),
                        'ajax_token' => $ajax_token,
                        'cart_token' => $cart_token,
                        'payment_check_token' => $payment_check_token,
                    ];
                }
            }
            if ($id_cart && $ajax_send) {
                if ($this->isSimplyInAuth()) {
                    $return['simplyin']['order'] = [
                        'id_cart' => $id_cart,
                        'order_token' => $this->getOrderToken($id_cart),
                    ];
                }
            }
        }
        if (Configuration::get('SIMPLYIN_PAYMENTS')) {
            $return['simplyin'][''] = [
                'check_payment_ajax' => $this->context->link->getModuleLink('simplyin', 'paymentajax'),
                'id_cart' => $this->context->cart->id,
                'id_module' => $this->id,
                'ajax_token' => $this->getAjaxToken(),
                'cart_token' => $this->getCartToken($this->context->cart->id),
                'payment_check_token' => $this->getPaymentCheckToken($this->getAjaxToken(), $this->getCartToken($this->context->cart->id)),
                'status' => SimplyInPayments::getPaymentInfo($this->context->cart->id),
            ];

        }
        if (!empty($override)) {
            $return = array_merge_recursive($return, $override);
        }

        return $return;
    }

    public function checkPaymentAjax()
    {
        $ajax_token = Tools::getValue('ajax_token');
        $cart_token = Tools::getValue('cart_token');
        $payment_check_token = Tools::getValue('payment_check_token');
        $id_cart = (int) Tools::getValue('id_cart');
        if ($ajax_token != $this->getAjaxToken() || $cart_token != $this->getCartToken($id_cart) || $payment_check_token != $this->getPaymentCheckToken($ajax_token, $cart_token)) {
            return [
                'status' => 'error',
                'message' => 'Token error',
            ];
        } else {
            $status = SimplyInPayments::getPaymentInfo($id_cart);
            if ($status) {
                return [
                    'status' => $status->status,
                    'date' => $status->date_upd
                ];
            }
        }
    }

    public function addNewOrderState($state, $names)
    {
        if (!(Validate::isInt(Configuration::get($state)) && Validate::isLoadedObject($order_state = new OrderState(Configuration::get($state))))) {
            $order_state = new OrderState();

            if (!empty($names)) {
                foreach ($names as $code => $name) {
                    $order_state->name[Language::getIdByIso($code)] = $name;
                }
            }
            $order_state->send_email = false;
            $order_state->invoice = false;
            $order_state->logable = false;
            $order_state->unremovable = true;
            $order_state->color = '#002124';
            $order_state->module_name = 'simplyin';

            if (!$order_state->add() || !Configuration::updateValue($state, $order_state->id)) {
                return false;
            }
            copy(_PS_MODULE_DIR_ . $this->name . '/logo.gif', _PS_IMG_DIR_ . 'os/' . $order_state->id . '.gif');

        }

        return $order_state->id;
    }

    public function installOrderStates()
    {
        $this->addNewOrderState('SIMPLYIN_PAYMENT_STATUS_PENDING', ['en' => 'SimplyIN - Payment pending', 'pl' => 'SimplyIN - Oczekiwanie na płatność']);
        $this->addNewOrderState('SIMPLYIN_PAYMENT_STATUS_CANCELED', ['en' => 'SimplyIN - Payment canceled', 'pl' => 'SimplyIN - Płatność anulowana']);
        return true;
    }

    public function hookActionAdminOrdersTrackingNumberUpdate($params)
    {
        if (Configuration::get('SIMPLYIN_CONNECTED')) {
            $order = $params['order'];

            if ($order->id && !empty($order->shipping_number)) {
                $order_reference = $order->reference;

                $tracking_number = $order->shipping_number;

                $customer = $order->getCustomer();
                $order_email = $customer->email;

                if (empty($order_email)) {
                    return;
                }

                $tracking_number = ['number' => $order->shipping_number];

                $newOrderStatus = '';

                $this->api->sendEncryptedData(
                    $order_email,
                    $order_reference,
                    $newOrderStatus,
                    $tracking_number
                );
            }
        }
    }

    public function getBackendUrl()
    {
        return $this->backend_url;
    }

    public function simplyInUpdateMerchantSettings($new_api_key = '', $admin = false)
    {
        if (self::$simply_update_settings_called) {
            return true;
        }


        if (empty($new_api_key)) {
            $new_api_key = Configuration::get('SIMPLYIN_SECRET_KEY');
        }

        if (empty($new_api_key)) {
            return false;
        }

        $last_update_date = Configuration::get('SIMPLYIN_LAST_UPDATE_INFO');

        if (!$admin && $last_update_date) {
            $last_update_timestamp = strtotime($last_update_date);
            if ($last_update_timestamp > (time() - 3600)) {
                return true;
            }
        }

        self::$simply_update_settings_called = true;

        $params = [
            'platform' => 'PrestaShop',
            'pluginVersion' => $this->version,
            'shopName' => Configuration::get('PS_SHOP_NAME'),
        ];

        $response = $this->api->sendRequest('plugin-sync/info', 'GET', null, $params, $new_api_key);

        if (!$response || isset($response->code)) {
            return false;
        }

        Configuration::updateValue('SIMPLYIN_LAST_UPDATE_INFO', date('Y-m-d H:i:s'));

        if (!Tools::isSubmit('submitSimplyinModule') && isset($response->switchInfo->isChecked)) {
            Configuration::updateValue('SIMPLYIN_SAVE_CHECKBOX', $response->switchInfo->isChecked);
        }

        if (isset($response->transactionHistoryLengthDays)) {
            Configuration::updateValue('SIMPLYIN_TRANS_HISTORY_LENGTH_DAY', $response->transactionHistoryLengthDays);
        }

        if (isset($response->switchInfo->lastChange)) {
            Configuration::updateValue('SIMPLYIN_SWITCH_LASTCHANGE', $response->switchInfo->lastChange);
        }

        if (isset($response->logisticsPartnersOrder->order)) {
            Configuration::updateValue('SIMPLYIN_LOGISTICSPARTNERSORDER', json_encode($response->logisticsPartnersOrder->order));
        }

        if (!empty($response->docsInfo) && is_array($response->docsInfo)) {
            $docs = [];
            foreach ($response->docsInfo as $doc) {
                if (isset($doc->key)) {
                    $docs[$doc->key] = $doc->url;
                }
            }
            if (!empty($docs)) {
                Configuration::updateValue('SIMPLYIN_DOCS_INFO', json_encode($docs));
            }
        }

        return true;
    }


    public function hookActionValidateOrder($params)
    {
        if (Configuration::get('SIMPLYIN_CONNECTED')) {
            SimplyInOrderSync::addOrderByParams($params);
        }
    }

    public function checkUserExist($user_email)
    {
        return SimplyInEmailCache::checkUserExist($user_email);
    }

    public function getCronSecureKey()
    {
        return md5(_COOKIE_KEY_ . '-getCronSecureKey');
    }

    public function execCron()
    {
        if (Tools::getValue('secure_key') !== $this->getCronSecureKey()) {
            exit('Invalid secure key');
        }
        SimplyInOrderSync::refreshOrdersSync();
        SimplyInOrderSync::synchroneOrders();
    }

    public function hookActionCarrierUpdate($params)
    {
        $carriers = $this->getCarriersKey();
        $carrier = $params['carrier'];
        $carrier_id = $carrier->id;
        $carrier_name = $carrier->name;
        $carrier_key = '';
        foreach ($carriers as $key) {
            if ($this->getConfigFormValues()[$key] == $carrier_id) {
                $carrier_key = $key;
                break;
            }
        }
        if ($carrier_key) {
            $config = $this->getConfigFormValues();
            $config[$carrier_key] = 0;
            Configuration::updateValue($carrier_key, 0);
        }
    }

    public function getCartCarrierCost()
    {
        $carrierKeys = $this->getCarriersKey();
        $carriers = [];

        $carrierMap = [
            'SIMPLYIN_DPD_CARRIER_ID' => 'dpd',
            'SIMPLYIN_UPS_CARRIER_ID' => 'ups',
            'SIMPLYIN_DHL_CARRIER_ID' => 'dhl',
            'SIMPLYIN_FEDEX_CARRIER_ID' => 'fedex',
            'SIMPLYIN_INPOST_CARRIER_ID' => 'inpost',
            'SIMPLYIN_ORLEN_CARRIER_ID' => 'orlen',
            'SIMPLYIN_DEFAULT_CARRIER_ID' => 'default',
        ];

        foreach ($carrierKeys as $key) {
            if (array_key_exists($key, $carrierMap) && Configuration::get($key)) {
                $carriers[$carrierMap[$key]] = Configuration::get($key);
            }
        }

        $carrierCosts = [];

        foreach ($carriers as $key => $idCarrier) {
            if ($idCarrier) {
                $carrierCosts[$key] = $this->context->getCurrentLocale()->formatPrice(
                    $this->context->cart->getOrderTotal(
                        true, Cart::ONLY_SHIPPING, null, $idCarrier
                    ),
                    Currency::getIsoCodeById((int) $this->context->cart->id_currency)
                );
            }
        }

        return $carrierCosts;
    }

    public function getAjaxToken()
    {
        return md5($this->name . _COOKIE_KEY_ . $this->version);
    }

    public function getCartToken($id_cart = false)
    {
        if ($id_cart) {
            $id_cart = (int) $this->context->cart->id;
        }
        return md5($this->name . _COOKIE_KEY_ . $this->version . $id_cart.'-cart-token');
    }

    public function getCustomerEmail()
    {
        $simplyin_email = $this->simplyin_session->get('simplyin_email');
        $simplyin_auth_token = $this->simplyin_session->get('simplyin_auth_token');
        if (isset($simplyin_email) && isset($simplyin_auth_token)) {
            $email = $simplyin_email;
            $refresh_token = $simplyin_auth_token;
            if (!empty($email) && !empty($refresh_token)) {
                return $email;
            }
        }

        return '';
    }

    private function updateDeliveryaddress($id_address_delivery, $id_cart)
    {
        $sql = '
            UPDATE      `' . _DB_PREFIX_ . 'cart_product`
            SET         id_address_delivery = ' . (int) $id_address_delivery . '
            WHERE       id_cart = ' . (int) $id_cart;

        Db::getInstance()->execute($sql);
        $sql = '
            UPDATE      `' . _DB_PREFIX_ . 'customization`
            SET         id_address_delivery = ' . (int) $id_address_delivery . '
            WHERE       id_cart = ' . (int) $id_cart;
        Db::getInstance()->execute($sql);
    }

    public function hookActionCustomerAccountUpdate($params)
    {
        if (Tools::isSubmit('submitCreate') && Tools::isSubmit('controller') == 'order') {
            $id_address_invoice = $this->simplyin_session->get('simplyin_id_address_invoice');
            if (isset($id_address_invoice) && !empty($id_address_invoice)) {
                $this->context->cart->id_address_invoice = $id_address_invoice;
            }
            $id_address_delivery = $this->simplyin_session->get('simplyin_id_address_delivery');

            if (isset($id_address_delivery) && !empty($id_address_delivery)) {
                $this->context->cart->id_address_delivery = $id_address_delivery;
                $this->context->cart->save();
            }
            $this->updateDeliveryaddress($id_address_delivery, $this->context->cart->id);
            $checkout_session_data = $this->getCheckoutSessionData();
            $this->upgradeSessionData($checkout_session_data);
        }
    }

    public function getCheckoutSessionData()
    {
        $sql = '
            SELECT      checkout_session_data
            FROM        `' . _DB_PREFIX_ . 'cart`
            WHERE       id_cart = ' . (int) $this->context->cart->id;

        $checkout_session_data = Db::getInstance()->getValue($sql);
        return json_decode($checkout_session_data, true);
    }

    public function upgradeSessionData($data, $redirect = true)
    {
        $opc = $this->getOPC();
        if ($opc == 'thecheckout') {
            $this->context->cookie->addreses_reset_at_cart_id = $this->context->cart->id;
            $this->context->cookie->write();
        }
        $checksum = new CartChecksum(new AddressChecksum());
        $id_address_invoice = $this->simplyin_session->get('simplyin_id_address_invoice');
        $id_address_delivery = $this->simplyin_session->get('simplyin_id_address_delivery');
        if (!empty($id_address_invoice) && !empty($id_address_delivery)) {
            $data = [
                'checkout-personal-information-step' => [
                    'step_is_reachable' => 1,
                    'step_is_complete' => 1,
                ],
                'checkout-addresses-step' => [
                    'step_is_reachable' => 1,
                    'step_is_complete' => 1,
                    'use_same_address' => (int) ($id_address_invoice == $id_address_delivery),
                ],
                'checkout-delivery-step' => [
                    'step_is_reachable' => 1,
                    'step_is_complete' => 1,
                ],
                'checkout-payment-step' => [
                    'step_is_reachable' => 1,
                    'step_is_complete' => 0,
                ],
            ];
            $data['checksum'] = $checksum->generateChecksum($this->context->cart);

            $sql = '
                UPDATE      `' . _DB_PREFIX_ . 'cart`
                SET         checkout_session_data = \'' . pSQL(json_encode($data)) . '\'
                WHERE       id_cart = ' . (int) $this->context->cart->id;
            Db::getInstance()->execute($sql);
            if ($redirect) {
                Tools::redirect($this->context->link->getPageLink('order', true, null));
            }
        }
    }

    public function hookActionCustomerLogoutBefore($params)
    {
        if ($params['customer']->is_guest == false) {
            $this->logout();
        }
    }

    public function logout()
    {
        $this->simplyin_session->clearSession();
    }

    public function getOrderToken($id_cart)
    {
        return md5($this->name . _COOKIE_KEY_ . $id_cart);
    }

    public function getNewAccountPhone()
    {
        $new_account_phone = $this->simplyin_session->get('new_account_phone');
        if (!isset($new_account_phone) || empty($new_account_phone)) {
            return false;
        } else {
            return $new_account_phone;
        }
    }

    public function isSimplyInAuth()
    {
        $simplyin_auth_token = $this->simplyin_session->get('simplyin_auth_token');
        $new_account_phone = $this->simplyin_session->get('new_account_phone');
        if (isset($simplyin_auth_token) && !empty($simplyin_auth_token)) {
            return true;
        } else {
            if (isset($new_account_phone) && !empty($new_account_phone)) {
                return true;
            } else {
                return false;
            }
        }

        return false;
    }

    public function getLogos($methods) {
        if (!empty($methods)) {
            foreach ($methods as $method) {
                $url =  $method->paymentProvider->logoUrl;
                $localFile = dirname(__FILE__).'/views/img/provaiders/'.basename($url);
                if (!file_exists($localFile))
                {

                    $ch = curl_init($url);
                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
                    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0');
                    $result = curl_exec($ch);
                    $httpcode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

                    if ($httpcode == 200) {
                        file_put_contents($localFile, $result);
                    }

                    curl_close($ch);
                } else {
                    return 'provaiders/'.basename($url);
                }
            }
        }
    }

    public function hookPaymentOptions($params)
    {
        if (!$this->checkCurrency($params['cart'])) {
            return [];
        }
        if (!$this->active || !Configuration::get('SIMPLYIN_PAYMENTS')) {
            return [];
        }
        $currency_iso = $this->context->currency->iso_code;
        try {
            $options = [];
            $methods = $this->api->getPaymentMethods();

            if ($currency_iso != 'PLN') {
                foreach ($methods as $key => $method) {
                    if ($method->typeName == 'BLIK') {
                        unset($methods[$key]);
                    }
                }
            }
            $methods_by_provaider = [];
            foreach ($methods as $method) {
                $provaider = $method->paymentProvider->id;
                $methods_by_provaider[$provaider][] = $method;
            }

            if (!empty($methods_by_provaider)) {
                foreach ($methods_by_provaider as $key => $provaider_method) {
                    $this->smarty->assign(
                        $this->getTemplateVarInfos($provaider_method)
                    );
                    if (($img_override = $this->getLogos($provaider_method))) {
                        $img = $img_override;
                    }

                    $payment_option = new PaymentOption();
                    $payment_option->setModuleName($this->name)
                        ->setCallToActionText($this->l('Simple and Safe payments'))
                        ->setAction($this->context->link->getModuleLink($this->name, 'validation', [], true))
                        ->setAdditionalInformation($this->fetch('module:simplyin/views/templates/hook/simplyin_payment_info.tpl'));
                    if ($img) {
                        $payment_option->setLogo(Media::getMediaPath($this->_path . 'views/img/' . $img));
                    }

                    $options[] = $payment_option;
                }
                return $options;
            }
        } catch (Exception $exp) {
            return [];
        }
        return [];
    }

    public function getCurrentUrl() {
        return Tools::getCurrentUrlProtocolPrefix() . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];
    }

    public function hookDisplayOrderConfirmation($params)
    {
        if (!$this->active || !Configuration::get('SIMPLYIN_PAYMENTS')) {
            return;
        }

        $id_cart = $params['order']->id_cart;
        $payment_info = SimplyInPayments::getPaymentInfo($id_cart);
        if (!isset($params['customer']) || $params['customer']->is_guest) {
            $this->simplyin_session->clearSession();
        }

        if ($payment_info === false) {
            return;
        }

        $this->smarty->assign([
            'payment_info' => $payment_info,
            'current_url' => $this->getCurrentUrl(),
            'module_path' => $this->_path,
        ]);

        return $this->fetch('module:simplyin/views/templates/hook/payment_return.tpl');
    }

    public function checkCurrency($cart)
    {
        $currency_order = new Currency($cart->id_currency);
        $currencies_module = $this->getCurrency($cart->id_currency);

        if (is_array($currencies_module)) {
            foreach ($currencies_module as $currency_module) {
                if ($currency_order->id == $currency_module['id_currency']) {
                    return true;
                }
            }
        }

        return false;
    }

    public function getTemplateVarInfos($methods)
    {
        $methods = (array) $methods;
        $cart = $this->context->cart;
        $total = sprintf(
            $this->l('%1$s (tax incl.)'),
            $this->context->getCurrentLocale()->formatPrice($cart->getOrderTotal(true, Cart::BOTH), $this->context->currency->iso_code)
        );

        $add_card = false;
        $blik = false;
        $cards = false;
        foreach ($methods as $method) {
            if ($method->type == 'add_card') {
                $add_card = $method;
            } elseif ($method->type == 'blik') {
                $blik = $method;
            } else if ($method->type == 'card') {
                if ($cards === false) {
                    $cards = [];
                }
                $cards[] = $method;
            }
        }

        return [
            'total' => $total,
            'add_card' => $add_card,
            'blik' => $blik,
            'cards' => $cards
        ];
    }

    public function getPaymentCheckToken($ajax_token, $cart_token)
    {
        return md5($this->name . _COOKIE_KEY_ . $this->version . $ajax_token . $cart_token);
    }

    public function upgradeModule($version)
    {
        if ($version == '2.0.2') {
            SimplyInPayments::upgrade202();
        }
        else if ($version == '1.3.0') {
            $this->installOrderStates();
            SimplyInPayments::installSql();
            return true;
        } else if ($version == '1.2.9') {
            $this->registerHook('paymentOptions');
            $this->registerHook('paymentReturn');

            return true;
        } elseif ($version === '1.1.8') {
            $this->registerHook('actionCustomerAccountUpdate');
            $this->registerHook('actionCustomerLogoutBefore');

            return true;
        } elseif ($version === '1.1.3') {
            SimplyInOrderSync::installSql();
            SimplyInEmailCache::installSql();

            return true;
        } elseif ($version === '1.1.2') {
            SimplyInOrder::installSql();

            return true;
        } elseif ($version === '1.1.1') {
            $this->registerHook('actionAdminOrdersTrackingNumberUpdate');

            return true;
        } elseif ($version === '1.1.0') {
            $this->unregisterHooks();
            $this->registerHooks();
            SimplyInPoint::installSql();
            $sql = 'DROP TABLE IF EXISTS `' . _DB_PREFIX_ . 'simplyin`';
            Db::getInstance()->execute($sql);

            return true;
        } else {
            return true;
        }
    }

    public function getCheckout()
    {
        $thecheckout = Module::getInstanceByName('thecheckout');
        if ($thecheckout && $thecheckout->active) {
            return 'thecheckout';
        } else {
            return 'classic';
        }
    }

    public function updateOrderState($order, $new_state, $callback)
    {
        if ($order->current_state != (int)$new_state) {
            $history = new OrderHistory();
            $history->id_order = (int)$order->id;
            $history->id_employee = 0;
            $history->changeIdOrderState((int)$new_state, $order->id);
            $history->add();

        }
        if (method_exists('SimplyInPayments', $callback)) {
            SimplyInPayments::$callback($order->id_cart);
        }
    }

    public function loadObjects()
    {
        $this->merchand_api_key = Configuration::get('SIMPLYIN_SECRET_KEY');
        $this->api = SimplyInApi::getInstance();
        if (isset($this->merchand_api_key) && !empty($this->merchand_api_key)) {
            $this->api->setMerchantApiKey($this->merchand_api_key);
            $this->simplyInUpdateMerchantSettings();
        }
        $this->simplyin_session = SimplyInSession::getInstance();
    }
    // Do not remove translations for objects
    // $this->l('blik');
    // $this->l('card');
    // $this->l('visa');
    // $this->l('mastercard');
    // $this->l('amex');
    // $this->l('discover');
    // $this->l('diners');
}