<?php
/**
 * Copyright 2024-2025 Simply.IN Sp. z o.o.
 *
 * NOTICE OF LICENSE
 *
 * Licensed under the EUPL-1.2 or later.
 * You may not use this work except in compliance with the Licence.
 *
 * Copy of the Licence is available at:
 * https://joinup.ec.europa.eu/software/page/eupl
 * It is bundled with this package in the file LICENSE.txt
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the Licence is distributed on an as is basis,
 * without warranties or conditions of any kind, either express or implied.
 * Check the Licence for the specific language governing permissions
 * and limitations under the License.
 *
 * @author   Simply.IN Sp. z o.o.
 * @copyright 2024-2025 Simply.IN Sp. z o.o.
 * @license   https://joinup.ec.europa.eu/software/page/eupl
 */

class SimplyInOrderSync extends ObjectModel
{
    public $id;
    public $cart_id;
    public $order_id;
    public $is_synced = false;
    public $ignored = false;
    public $date_synced = '0000-00-00 00:00:00';
    /** @var $module instance of SimplyIn */
    private static $module = false;

    public static $token = '';
    public static $definition = array(
        'table' => 'simplyin_order_sync',
        'primary' => 'id_simplyin_order_sync',
        'multilang' => false,
        'multilang_shop' => false,
        'fields' => array(
            'cart_id' => array(
                'type' => self::TYPE_INT,
                'size' => 100,
                'index' => true,
            ),
            'order_id' => array(
                'type' => self::TYPE_INT,
                'size' => 36,
                'index' => true,
            ),
            'is_synced' => array(
                'type' => self::TYPE_BOOL,
            ),
            'ignored' => array(
                'type' => self::TYPE_BOOL,
            ),
            'date_synced' => array(
                'type' => self::TYPE_DATE,
            ),
        ),
    );

        public static function installSql()
    {
        $sql = [];
        $sql = '
            CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . self::$definition['table'] . '`
            (
                %fields%
            ) CHARACTER SET utf8 COLLATE utf8_general_ci;
        ';
        $fields_lang = [];
        $fields = [];
        $fields[] = self::$definition['primary'] . ' INT(11) PRIMARY KEY AUTO_INCREMENT';
        $fields_lang[] = self::$definition['primary'] . ' INT(11)';
        $fields_lang[] = 'id_lang INT(11)';
        foreach (self::$definition['fields'] as $key => $field) {
            $fields_query = '';
            switch ($field['type']) {
                case self::TYPE_STRING:
                case self::TYPE_HTML:
                    if (isset($field['lang'])) {
                        if (isset($field['size'])) {
                            $fields_lang[] = $key . ' VARCHAR(' . $field['size'] . ')';
                        } else {
                            $fields_lang[] = $key . ' LONGTEXT';
                        }
                    } else {
                        if (isset($field['size'])) {
                            $fields_query = $key . ' VARCHAR(' . $field['size'] . ')';
                        } else {
                            $fields_query = $key . ' LONGTEXT';
                        }
                    }
                    break;
                case self::TYPE_INT:
                case self::TYPE_BOOL:
                    if (isset($field['size'])) {
                        $fields_query = $key . ' INT(' . $field['size'] . ')';
                    } else {
                        $fields_query = $key . ' INT(11)';
                    }
                    break;
                case self::TYPE_DATE:
                    $fields_query = $key . ' DATETIME';
                    break;
                case self::TYPE_FLOAT:
                    $fields_query = $key . ' FLOAT';
                    break;
            }
            $fields[] = $fields_query;
        }
        $fields = implode(',' . PHP_EOL . str_repeat(' ', 16), array_filter($fields, 'strlen'));
        if (sizeof($fields_lang) > 2) {
            $sql_lang = '
                CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . self::$definition['table'] . '_lang`
                (
                    %fields%
                ) CHARACTER SET utf8 COLLATE utf8_general_ci;
            ';
            $fields_lang = implode(',' . PHP_EOL . str_repeat(' ', 16), array_filter($fields_lang, 'strlen'));
            $sql_lang = str_replace('%fields%', $fields_lang, $sql_lang);
            Db::getInstance()->execute($sql_lang);
        }
        $sql = str_replace('%fields%', $fields, $sql);

        $return = Db::getInstance()->execute($sql);
        if ($return) {
            foreach (self::$definition['fields'] as $key => $field) {
                if (isset($field['index']) && $field['index']) {
                    $sql = 'ALTER TABLE `' . _DB_PREFIX_ . self::$definition['table'] . '` ADD INDEX (`' . $key . '`)';
                    $return &= Db::getInstance()->execute($sql);
                }
            }
        }

        return $return;
    }

    public static function uninstallSql()
    {
        $sql = [];
        $sql[] = 'DROP TABLE IF EXISTS `' . _DB_PREFIX_ . self::$definition['table'] . '`';
        $sql[] = 'DROP TABLE IF EXISTS `' . _DB_PREFIX_ . self::$definition['table'] . '_lang`';
        foreach ($sql as $item) {
            if (!Db::getInstance()->execute($item)) {
                return false;
            }
        }

        return true;
    }

    public static function addOrderByParams($params)
    {
        if (isset($params['order']) && isset($params['order']->id)) {
            $order = new SimplyInOrderSync();
            $order->cart_id = $params['cart']->id;
            $order->order_id = $params['order']->id;
            $order->is_synced = false;
            $order->add();
        }
    }

    public static function removeOldOrders()
    {
        $days = (int)Configuration::get('SIMPLYIN_TRANS_HISTORY_LENGTH_DAY');
        $sql = '
            DELETE      FROM `' . _DB_PREFIX_ . self::$definition['table'] . '`
            WHERE       order_id IN (
                SELECT  id_order
                FROM    `' . _DB_PREFIX_ . 'orders`
                WHERE   date_add < DATE_SUB(NOW(), INTERVAL ' . $days . ' DAY)
            )
        ';
        Db::getInstance()->execute($sql);
    }

    public static function refreshOrdersSync()
    {
        $days = (int)Configuration::get('SIMPLYIN_TRANS_HISTORY_LENGTH_DAY');
        $sql = '
            SELECT      o.`id_order` as order_id, o.`id_cart` as cart_id
            FROM        `' . _DB_PREFIX_ . 'orders` o
            WHERE       o.`date_add` >= DATE_SUB(NOW(), INTERVAL ' . $days . ' DAY)
            AND         o.id_order NOT IN
            (
                SELECT  order_id
                FROM    `' . _DB_PREFIX_ . self::$definition['table'] . '`
            )
        ';

        $orders = Db::getInstance()->executeS($sql);

        if ($orders) {
            foreach ($orders as $order) {
                $order_sync = new SimplyInOrderSync();
                $order_sync->cart_id = $order['cart_id'];
                $order_sync->order_id = $order['order_id'];
                $order_sync->is_synced = false;
                $order_sync->add();
            }
        }

        self::markIgnorredOrders();
    }

    public static function markIgnorredOrders()
    {
        $sql = '
            SELECT     o.`id_order`, c.`email`
            FROM       `' . _DB_PREFIX_ . 'orders` o
            JOIN       `' . _DB_PREFIX_ . 'customer` c
                ON     o.`id_customer` = c.`id_customer`
            WHERE      o.`id_order` IN
            (
                SELECT  `order_id`
                FROM    `' . _DB_PREFIX_ . self::$definition['table'] . '`
                WHERE   `is_synced` = 0
                AND     `ignored` = 0
            )
        ';
        $result = Db::getInstance()->executeS($sql);
        $orders_by_email = [];
        if (!empty($result)) {
            foreach ($result as $row) {
                $orders_by_email[$row['email']][] = $row['id_order'];
            }
            $list = SimplyInEmailCache::checkUserListExist(array_keys($orders_by_email));
            if (!empty($list)) {
                foreach ($list as $email => $exist) {
                    if (!$exist) {
                        foreach ($orders_by_email[$email] as $order_id) {
                            $order = SimplyInOrderSync::getByIdOrder($order_id);
                            $order->ignored = true;
                            $order->date_synced = Date('Y-m-d H:i:s');
                            $order->save();
                        }
                    }
                }
            }

        }
    }

    public static function synchroneOrders() {
        $sql = '
            SELECT      *
            FROM        `' . _DB_PREFIX_ . self::$definition['table'] . '`
            WHERE       is_synced = 0
            AND         ignored = 0
            ORDER BY    id_simplyin_order_sync ASC
            LIMIT       50
        ';
        $rows = Db::getInstance()->executeS($sql);
        foreach ($rows as $row) {
            self::createOrder($row['order_id']);
        }
    }

    public static function getModule()
    {
        if (self::$module === false) {
            $module = Module::getInstanceByName('simplyin');
            self::$module = $module;
        } else {
            $module = self::$module;
        }
        return self::$module;
    }

    public static function createOrder($order_id) {
        $module = self::getModule();
        $order = new Order($order_id);
        $customer = new Customer($order->id_customer);
        $user_email = $customer->email;
        if (empty($user_email)) {
            $object = SimplyInOrderSync::getByIdOrder($order_id);
            $object->ignored = true;
            $object->date_synced = Date('Y-m-d H:i:s');
            $object->save();
        }
        else if (SimplyInEmailCache::checkUserExist($user_email)) {
            $module = Module::getInstanceByName('simplyin');
            $plugin_version = $module->version;
            $shop_version = _PS_VERSION_;
            $items_data = self::getOrderItemsData($order);
            $payment_method = [
                'method' => $order->module,
                'title' => $order->payment,
            ];
            $parcel_machine_id = self::getParcelMachineIdByOrderId($order_id);

            $body_data = self::buildExistingAccountOrderData(
                $order,
                $items_data,
                $payment_method,
                $plugin_version,
                $shop_version,
                $user_email,
                $parcel_machine_id
            );

            $response = SimplyInApi::getInstance()->sendRequest('checkout/createOrder',"POST",$body_data);

            if (isset($response->message) && $response->message == 'Order already exists') {
                $object = SimplyInOrderSync::getByIdOrder($order_id);
                $object->is_synced = true;
                $object->date_synced = Date('Y-m-d H:i:s');
                $object->save();
            } else if (isset($response->ok)) {
                $object = SimplyInOrderSync::getByIdOrder($order_id);
                $object->is_synced = true;
                $object->date_synced = Date('Y-m-d H:i:s');
                $object->save();
            }
        } else {
            $object = SimplyInOrderSync::getByIdOrder($order_id);
            $object->ignored = true;
            $object->date_synced = Date('Y-m-d H:i:s');
            $object->save();
        }
    }

    public static function getByIdOrder($order_id)
    {
        $sql = '
            SELECT      *
            FROM        `' . _DB_PREFIX_ . self::$definition['table'] . '`
            WHERE       order_id = ' . (int)$order_id;
        $row = Db::getInstance()->getRow($sql);
        if ($row) {
            $object = new SimplyInOrderSync($row['id_simplyin_order_sync']);
            return $object;
        } else {
            $object = new SimplyInOrderSync();
            $object->order_id = $order_id;
            $order = new Order($order_id);
            $object->cart_id = $order->id_cart;
            return $object;
        }
    }

    public static function getOrderItemsData($order) {
        $items_data = [];
        $products = $order->getProducts();
        $currency = new Currency($order->id_currency);
        $context = Context::getContext();
        foreach ($products as $product) {
            $total_tax = 0;
            $tax_rate = 0;
            if ($product['unit_price_tax_excl'] > 0) {
                $tax_rate = Tools::ps_round($product['unit_price_tax_incl'] / $product['unit_price_tax_excl'], 2);
                $tax_rate = (int) ($tax_rate * 100 - 100);

                $total_tax = $product['unit_price_tax_incl'] - $product['unit_price_tax_excl'];
            } else {
                $total_tax = 0;
                $tax_rate = 0;
            }
            $url = $context->link->getProductLink($product['id_product']);
            $cover_image = Product::getCover($product['id_product']);
            if (isset($cover_image['id_image'])) {
                $image = $context->link->getImageLink(Tools::link_rewrite($product['product_name']), $cover_image['id_image'], ImageType::getFormattedName('small'));
            } else {
                $image = '';
            }
            $items_data[] = [
                'name' => $product['product_name'],
                'url' => $url,
                'price_net' => Tools::ps_round($product['unit_price_tax_excl'], 2),
                'price' => Tools::ps_round($product['unit_price_tax_incl'], 2),
                'quantity' => (int) $product['product_quantity'],
                'tax_amount' => Tools::ps_round($total_tax, 2),
                'tax_rate' => $tax_rate,
                'thumbnailUrl' => $image,
                'currency' => $currency->iso_code,
            ];
        }
        return $items_data;
    }

    public static function getIpAddressByOrderId($id_order)
    {
        $sql = '
            SELECT      c.`ip_address`
            FROM        `' . _DB_PREFIX_ . 'connections` c
            JOIN        `' . _DB_PREFIX_ . 'guest` g
                ON      c.id_guest = g.id_guest
            JOIN        `' . _DB_PREFIX_ . 'orders` o
                ON      g.id_customer = o.id_customer
            WHERE       o.id_order = ' . (int)$id_order;

        $ip_address = Db::getInstance()->getValue($sql);
        if (!empty($ip_address)) {
            return long2ip($ip_address);
        }
        else {
            return '';
        }
    }

    public static function buildExistingAccountOrderData(
        $order,
        $items_data,
        $payment_method,
        $plugin_version,
        $shop_version,
        $user_email,
        $parcel_machine_id
    ) {
        $billingData = self::getBillingData($order);
        $hostAddres= gethostname();
        $hostIp = gethostbyname($hostAddres);
        $order_ip_address = self::getIpAddressByOrderId($order->id);
        $currency = new Currency($order->id_currency);
        $body_data = [
            'newOrderData' => [
                'price' => (float) $order->total_paid_tax_incl,
                'shippingPrice' => $order->total_shipping_tax_incl,
                'currency' => $currency->iso_code,
                'items' => $items_data,
                'statusInfo' => self::getOrderStatusInfo($order),
                'trackingNumbers' => self::getTrackingNumbers($order),
                'shopOrderNumber' => $order->reference,
                'placedDuringAccountCreation' => false,
                'pluginVersion' => $plugin_version,
                'shopVersion' => $shop_version,
                'shopName' => Configuration::get('PS_SHOP_NAME'),
                'shopUserEmail' => $user_email,
                'userIp' => $order_ip_address,
                'shopIp' => $hostIp,
                'billingData' => $billingData,
                'payment_method' => $payment_method['method'],
                'payment_method_title' => $payment_method['title'],
            ],
            'pluginVersion' => $plugin_version,
            'shopVersion' => $shop_version,
            'shopName' => Configuration::get('PS_SHOP_NAME'),
            'shopUserEmail' => $user_email,
            'userIp' => $order_ip_address,
            'shopIp' => $hostIp,
        ];

        if (!empty($parcel_machine_id)) {
            $body_data["newOrderData"]["parcelLockerMinimalInfo"] = [
                "lockerId" => $parcel_machine_id,
                "providerName" => "inpost"
            ];
        } else {
            $shippingData = self::getShippingData($order);
            $body_data["newOrderData"]["shippingData"] = $shippingData;
        }
        $body_data['merchantApiKey'] = Configuration::get('SIMPLYIN_SECRET_KEY');

        return $body_data;
    }

    public static function getBillingData($order) {
        $address = new Address($order->id_address_invoice);

        $billingData = [
            "name" => trim($address->firstname),
            "surname" => trim($address->lastname),
            "street" => trim($address->address1),
            "appartmentNumber" => trim($address->address2),
            "city" => trim($address->city),
            "postalCode" => trim($address->postcode),
            "country" => Country::getNameById(Context::getContext()->language->id, $address->id_country),
            "state" => ($address->id_state ? State::getNameById($address->id_state) : ''),
            "companyName" => trim($address->company),
            "taxId" => trim($address->vat_number),
        ];

        return $billingData;
    }

    public static function getShippingData($order, $simply_shipping_id = '') {
        $address = new Address($order->id_address_delivery);

        $shippingData = [
            "icon" => "🏡",
            "addressName" => "",
            "name" => trim($address->firstname),
            "surname" => trim($address->lastname),
            "street" => trim($address->address1),
            "appartmentNumber" => trim($address->address2),
            "city" => trim($address->city),
            "postalCode" => trim($address->postcode),
            "country" => Country::getNameById(Context::getContext()->language->id, $address->id_country),
            "state" => ($address->id_state ? State::getNameById($address->id_state) : ''),
            "companyName" => trim($address->company),
        ];

        if (!empty($simply_shipping_id)) {
            $shippingData["_id"] = $simply_shipping_id;
        }

        return $shippingData;
    }

    public static function getOrderStatusInfo($order)
    {
        $order_state_id = $order->getCurrentState();

        $date = Date('Y-m-d\TH:i:s.u\Z', strtotime($order->date_upd));
        $order_state = new OrderState($order_state_id, Context::getContext()->language->id);
        $order_status = 'processing';
        return [
            'orderStatus' => $order_status,
            'statusDate' => $date
        ];
    }


    public static function getParcelMachineIdByOrderId($id_order)
    {
        // to do
        return '';
    }

    public static function getTrackingNumbers($order)
    {
        $tracking_numbers = [];
        if (!empty($order->shipping_number)) {
            $tracking_numbers[] = $order->shipping_number;
        }
        return $tracking_numbers;
    }

    public static function getOrderData($id_order)
    {
        $simplyin_session = SimplyInSession::getInstance();
        $order = new Order($id_order);
        $customer = new Customer($order->id_customer);
        $currency = new Currency($order->id_currency);
        $products = [];
        $orderProducts = $order->getProducts();
        $currencyISO = $currency->iso_code;
        foreach ($orderProducts as $product) {
            $productId = $product['product_id'];
            $productName = 'img';

            $productThumbnailId = Product::getCover($productId)['id_image'];
            $productObj = new Product($productId, false, Context::getContext()->language->id);
            $productDescription = strip_tags($productObj->description);
            $productLink = Context::getContext()->link->getProductLink($product);
            $thumbnailUrl = Context::getContext()->link->getImageLink($productName, $productThumbnailId, ImageType::getFormattedName('small'));
            $products[] = [
                'name' => $product['product_name'],
                'quantity' => (int) $product['product_quantity'],
                'price_net' => (float) $product['total_price_tax_excl'],
                'price' => (float) $product['total_price_tax_incl'],
                'productDescription' => $productDescription,
                'url' => $productLink,
                'thumbnailUrl' => $thumbnailUrl,
                'currency' => $currencyISO,
            ];
        }
        $language = new Language($order->id_lang);

        $billing = new Address($order->id_address_invoice);
        $delivery = new Address($order->id_address_delivery);

        $billingCountry = new Country($billing->id_country);
        $deliveryCountry = new Country($delivery->id_country);
        $billingState = new State($billing->id_state);
        $deliveryState = new State($delivery->id_state);
        $simplyin_billing = $simplyin_session->get('simplyin_billing', '');
        $simplyin_shipping = $simplyin_session->get('simplyin_shipping', '');
        $icon_billing = trim($billing->company) != '' ? '🏢' : '🏡';
        $icon_shipping = trim($delivery->company) != '' ? '🏢' : '🏡';

        $exploded = explode(' ', $billing->address1);
        if (!isset($exploded[1])) {
            $exploded[1] = '';
        }
        $billingData = [
            '_id' => $simplyin_billing,
            'icon' => $icon_billing,
            'addressName' => '',
            'street' => trim($exploded[0]),
            'appartmentNumber' => trim($exploded[1] != '' ? $exploded[1] : ''),
            'city' => trim($billing->city),
            'postalCode' => trim($billing->postcode),
            'country' => trim($billingCountry->iso_code),
            'companyName' => trim($billing->company),
            'name' => trim($billing->firstname),
            'surname' => trim($billing->lastname),
            'taxId' => trim($billing->vat_number),
            'state' => trim($billingState->iso_code ?? ''),
        ];

        $exploded = explode(' ', $delivery->address1);
        if (!isset($exploded[1])) {
            $exploded[1] = '';
        }

        $shippingData = [
            '_id' => $simplyin_shipping,
            'icon' => $icon_shipping,
            'addressName' => '',
            'street' => trim($exploded[0]),
            'appartmentNumber' => trim($exploded[1] != '' ? $exploded[1] : ''),
            'city' => trim($delivery->city),
            'postalCode' => trim($delivery->postcode),
            'country' => trim($deliveryCountry->iso_code),
            'companyName' => trim($delivery->company),
            'name' => trim($delivery->firstname),
            'surname' => trim($delivery->lastname),
            'state' => trim($deliveryState->iso_code ?? ''),
        ];

        $delivery_point = SimplyInPoint::getByIdCart($order->id_cart);
        $phoneNumber = $delivery->phone;
        $phone = $simplyin_session->get('new_account_phone', false);
        if ($phone) {
            $phoneNumber = $phone;
        }
        $payment_method = $order->module;
        $payment_method_title = $order->payment;

        $simplyin_payment_type = SimplyInSession::getInstance()->get('simplyin_payment_type', false);
        if ($simplyin_payment_type)
        {
             $payment_method = $simplyin_payment_type;
             $module = self::getModule();
             $payment_method = $module->l($payment_method);
        }
        $simplyin_payment_provaider = SimplyInSession::getInstance()->get('simplyin_payment_provaider', false);
        if ($simplyin_payment_provaider)
        {
            $payment_method_title = $simplyin_payment_provaider;
        }

        $orderData = [
            'newAccountData' => [
                'name' => $delivery->firstname,
                'surname' => $delivery->lastname,
                'phoneNumber' => $phoneNumber,
                'email' => strtolower($customer->email),
                'language' => strtoupper(substr($language->language_code, 0, 2)),
                'marketingConsent' => false,
            ],
            'newOrderData' => [
                'shopOrderNumber' => $order->reference,
                'price' => (float) $order->total_paid,
                'shippingPrice' => (float) $order->total_shipping,
                'currency' => $currency->iso_code,
                'items' => $products,
                'placedDuringAccountCreation' => true,
                'billingData' => $billingData,
                'shippingData' => $shippingData,
                'payment_method' => $payment_method,
                'payment_method_title' => $payment_method_title,
            ],
        ];

        if ($delivery_point->id) {
            unset($orderData['newOrderData']['shippingData']);
            $orderData['newOrderData']['parcelLockerMinimalInfo'] = [
                'providerName' => $delivery_point->label,
                'lockerId' => $delivery_point->lockerId,
              ];
        }

        return $orderData;
    }

}
