<?php
/**
 * 2007-2020 PrestaShop and Contributors
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License 3.0 (AFL-3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/AFL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * @author    PrestaShop SA <contact@prestashop.com>
 * @copyright 2007-2020 PrestaShop SA and Contributors
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License 3.0 (AFL-3.0)
 * International Registered Trademark & Property of PrestaShop SA
 */

/**
 * @since 1.5.0
 *
 * @property Ps_Wirepayment $module
 */

class SimplyInValidationModuleFrontController extends ModuleFrontController
{
    public $account_created = false;
    public $simplyin_session = false;

    public function postProcess()
    {
        $this->simplyin_session = SimplyInSession::getInstance();
        $cart = $this->context->cart;

        if ($cart->id_customer == 0 || $cart->id_address_delivery == 0 || $cart->id_address_invoice == 0 || !$this->module->active) {
            Tools::redirect('index.php?controller=order&step=1');
        }

        if (!$cart->id) {
            Tools::redirect('index.php?controller=order&step=1');
        }

        $authorized = false;
        foreach (Module::getPaymentModules() as $module) {
            if ($module['name'] == 'simplyin') {
                $authorized = true;
                break;
            }
        }

        if (!Configuration::get('SIMPLYIN_PAYMENTS')) {
            $authorized = false;
        }

        if (!$authorized) {
            exit($this->module->getTranslator()->trans('This payment method is not available.', [], 'Modules.Wirepayment.Shop'));
        }

        $customer = new Customer($cart->id_customer);
        if (!Validate::isLoadedObject($customer)) {
            Tools::redirect('index.php?controller=order&step=1');
        }

        $currency = $this->context->currency;
        $total = (float) $cart->getOrderTotal(true, Cart::BOTH);

        $this->module->validateOrder(
            $cart->id,
            (int) Configuration::get('SIMPLYIN_PAYMENT_STATUS_PENDING'),
            0,
            $this->module->displayName,
            null,
            [],
            (int) $currency->id,
            false,
            $customer->secure_key
        );

        $order = Order::getByCartId($cart->id);

        $this->placeOrderSimplyIn($cart, $order);
        $this->module->updateOrderState($order, (int) Configuration::get('SIMPLYIN_PAYMENT_STATUS_PENDING'), 'markAsPending');

        $redirect_url = 'index.php?controller=order-confirmation&id_cart=' . $cart->id .
                        '&id_module=' . $this->module->id .
                        '&id_order=' . $this->module->currentOrder .
                        '&key=' . $customer->secure_key;

        $this->createPaymentSimplyIn($cart, $redirect_url);
        Tools::redirect($redirect_url);
    }

    public function placeOrderSimplyIn($cart, $order)
    {
        $new_account_phone = $this->simplyin_session->get('new_account_phone', false);
        $auth_token = $this->getAuthToken();

        if (!empty($auth_token)) {
            $this->createOrderWithoutaccount($order);
            $this->simplyin_session->set('simplyin_auth_token', $auth_token);
        } elseif (!empty($new_account_phone)) {
            $this->createOrderAndAccount($order);
            $this->account_created = true;
            $this->simplyin_session->unset('new_account_phone', '');
        }
    }

    public function createOrderWithoutaccount($order)
    {
        $orderData = SimplyInOrderSync::getOrderData($order->id);
        $orderData['newOrderData']['placedDuringAccountCreation'] = false;
        unset($orderData['newAccountData']);

        return SimplyInApi::getInstance()->apiCall(
            $orderData,
            'POST',
            'checkout/createOrderWithoutaccount',
            $this->getAuthToken()
        );
    }

    public function createOrderAndAccount($order)
    {
        $customer = new Customer($order->id_customer);
        SimplyInEmailCache::clearCacheByEmail($customer->email);

        $orderData = SimplyInOrderSync::getOrderData($order->id);
        $orderData['newOrderData']['placedDuringAccountCreation'] = true;

        return SimplyInApi::getInstance()->apiCall($orderData, 'POST', 'checkout/createOrderAndAccount');
    }

    public function getAuthToken()
    {
        return $this->simplyin_session->get('simplyin_auth_token', false);
    }

    public function createPaymentSimplyIn($cart, $redirect_url)
    {
        $redirect_url = Tools::getShopDomainSsl(true, true) . __PS_BASE_URI__ . $redirect_url;
        $order = Order::getByCartId($cart->id);

        $payment_method_id = $this->simplyin_session->get('simplyin_payment_id');
        $payment_type = $this->simplyin_session->get('simplyin_payment_type');
        $title = $order->reference;
        $order_total = $order->getOrdersTotalPaid();
        $firstName = $this->context->customer->firstname;
        $lastName = $this->context->customer->lastname;
        $currency = Currency::getIsoCodeById((int) $order->id_currency);

        $simplyin_auth_token = $this->simplyin_session->get('simplyin_auth_token');
        $auth_token = $simplyin_auth_token;
        $paymentId = SimplyInPayments::getUUID($order);
        $email = !empty($simplyin_auth_token) ? $this->module->getCustomerEmail() : $this->context->customer->email;

        if ($payment_type == 'blik') {
            $paymentMethodData = [
                'blikCode' => $this->simplyin_session->get('simplyin_payment_blik'),
            ];
        } elseif ($payment_type == 'card') {
            $paymentMethodData = [
                'cardName' => $this->simplyin_session->get('simplyin_card_name'),
                'LastDigitsOfCardNumber' => $this->simplyin_session->get('simplyin_card_last_digits'),
                'cardProvider' => $this->simplyin_session->get('simplyin_card_provider'),
                'OTT' => $this->simplyin_session->get('simplyin_card_ott') ?: '',
                'saveCard' => false,
            ];
        } elseif ($payment_type == 'add-card') {
            if (!empty($simplyin_auth_token) || $this->account_created) {
                $card_name = $this->simplyin_session->get('simplyin_card_name');
                $save_card = (bool) $this->simplyin_session->get('simplyin_card_save');
            } else {
                $card_name = '';
                $save_card = false;
            }

            $paymentMethodData = [
                'cardName' => $card_name,
                'LastDigitsOfCardNumber' => $this->simplyin_session->get('simplyin_card_last_digits'),
                'cardProvider' => $this->simplyin_session->get('simplyin_card_provider'),
                'OTT' => $this->simplyin_session->get('simplyin_card_ott') ?: '',
                'saveCard' => $save_card,
            ];
        }

        $body = [
            'paymentId' => $paymentId,
            'paymentMethodId' => $payment_method_id,
            'title' => $title,
            'amount' => (float) $order_total,
            'currency' => $currency,
            'customerPaymentData' => [
                'email' => $email,
                'firstName' => $firstName,
                'lastName' => $lastName,
                'phone' => '',
            ],
            'paymentMethodData' => $paymentMethodData,
            'returnUrl' => $redirect_url,
            'webhookUrl' => $this->context->link->getModuleLink($this->module->name, 'webhook', [], true),
        ];

        $response = SimplyInApi::getInstance()->sendRequest(
            'payments/pay',
            'POST',
            $body,
            [],
            '',
            false,
            $auth_token
        );



        if (isset($response->code) && $response->code == 'FORBIDDEN') {
            $status = 'FAILD';
        } elseif (isset($response->status)) {
            $status = $response->status;
        } else {
            $status = 'FAILD';
        }

        $payment_provaider = $this->simplyin_session->get('simplyin_payment_provaider');
        $card_provaider = $this->simplyin_session->get('simplyin_card_provider');
        $provaider_image = $this->simplyin_session->get('simplyin_payment_image');
        $image = $this->simplyin_session->get('simplyin_payment_logo');

        $this->simplyin_session->clearSession();


        SimplyInPayments::addByCart(
            $cart,
            $payment_type,
            $status,
            $paymentId,
            $payment_provaider,
            $card_provaider,
            $provaider_image,
            $image
        );

        if ($status == 'FAILD') {
            $this->module->updateOrderState($order, (int) Configuration::get('SIMPLYIN_PAYMENT_STATUS_CANCELED'), 'markAsRejected');
        }

        if ($payment_type == 'card' || $payment_type == 'add-card') {
            if (isset($response->redirectUrl)) {
                header('Location: ' . $response->redirectUrl);
                session_write_close();
                exit;
            }
        }
    }
}
