<?php
/**
 * Copyright 2024-2025 Simply.IN Sp. z o.o.
 *
 * NOTICE OF LICENSE
 *
 * Licensed under the EUPL-1.2 or later.
 * You may not use this work except in compliance with the Licence.
 *
 * Copy of the Licence is available at:
 * https://joinup.ec.europa.eu/software/page/eupl
 * It is bundled with this package in the file LICENSE.txt
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the Licence is distributed on an as is basis,
 * without warranties or conditions of any kind, either express or implied.
 * Check the Licence for the specific language governing permissions
 * and limitations under the License.
 *
 * @author   Simply.IN Sp. z o.o.
 * @copyright 2024-2025 Simply.IN Sp. z o.o.
 * @license   https://joinup.ec.europa.eu/software/page/eupl
 */

require_once dirname(__FILE__) . '/../../models/SimplyInPoint.php';
require_once dirname(__FILE__) . '/../../classes/SimplyInApi.php';
require_once dirname(__FILE__) . '/../../classes/SimplyInSession.php';

class SimplyinAjaxModuleFrontController extends ModuleFrontController
{
    /** @var SimplyInApi|null This is an instance of SimplyInApi */
    public $api = false;
    /** @var SimplyInSession|null This is an instance of Simplyin */
    public $module = false;
    /** @var SimplyInSession|null This is an instance of SimplyInSession */
    public $simplyin_session = false;
    public $merchand_api_key = false;
    public $warnings = [];

    private $allowed_types = [
        'autocomplete',
        'confirmation',
        'delivery',
        'sms',
    ];

    public function __construct()
    {
        parent::__construct();
        $this->module = Module::getInstanceByName('simplyin');
        $this->simplyin_session = SimplyInSession::getInstance();
        $this->api = SimplyInApi::getInstance();
    }

    public function initContent()
    {
        parent::initContent();
    }

    public function displayAjax()
    {
        // header('Content-Type: application/json');
        $this->checkAjaxToken();
        if (Tools::isSubmit('action')) {
            $action = Tools::getValue('action');
            $method = $action . 'AjaxAction';

            if (method_exists($this, $method)) {
                exit(json_encode($this->{$method}()));
            } else {
                exit(json_encode(['error' => 'Invalid action']));
            }
        } else {
            exit(json_encode(['error' => 'No action specified']));
        }
    }

    public function getFormAjaxAction()
    {
        $type = Tools::getValue('type');
        if (!in_array($type, $this->allowed_types)) {
            return ['error' => 'Invalid form type'];
        }

        return $this->renderAjaxForm($type);
    }

    public function setNotificationId($email, $response)
    {
        $notification_token_id = isset($response['notificationTokenId']) ? $response['notificationTokenId'] : false;
        if ($notification_token_id) {
            $this->simplyin_session->set('simplyin_notifiaction_token_id', $notification_token_id);
        }
        $this->simplyin_session->set('simplyin_email', $email);
    }

    public function submitEmailAjaxAction()
    {
        $email = Tools::getValue('email');
        $logout = (int) Tools::getValue('logout');
        $login = (int) Tools::getValue('login');
        if ($login) {
            $customer = new Customer();
            $customer->getByEmail($email);
            if ($customer->id && !$customer->is_guest && $customer->active) {
            } else {
                return ['code' => 'NOT_FOUND'];
            }
        }
        if ($logout) {
            $this->module->logout();
        }
        $response = $this->api->apiCall(['email' => $email], 'POST', 'checkout/submitEmail');
        if ($response['code'] == 'NOT_FOUND' || $response['code'] == 'BAD_REQUEST') {
            $this->module->logout();

            return ['error' => $response['code']];
        } else {
            $this->setNotificationId($email, $response);

            return [
                'success' => true,
                'html' => $this->fetchPopupConfirmation($response),
                'show_popup' => true,
                'userUsedPushNotifications' => isset($response['userUsedPushNotifications']) ? $response['userUsedPushNotifications'] : '',
            ];
        }
    }

    public function sendSmsCodeAjaxAction()
    {
        $email = Tools::getValue('email');
        $response = $this->api->apiCall(['email' => $email, 'forceSms' => true], 'POST', 'checkout/submitEmail');

        if ($response['code'] == 'NOT_FOUND') {
            return ['error' => $response['code']];
        } else {
            $this->setNotificationId($email, $response);

            return [
                'success' => true,
                'html' => $this->fetchPopupConfirmation($response),
                'show_popup' => true,
                'userUsedPushNotifications' => isset($response['userUsedPushNotifications']) ? $response['userUsedPushNotifications'] : '',
            ];
        }
    }

    public function sendEmailCodeAjaxAction()
    {
        $email = Tools::getValue('email');
        $response = $this->api->apiCall(['email' => $email], 'POST', 'checkout/resend-checkout-code-via-email');
        if ($response['code'] == 'NOT_FOUND') {
            return ['error' => $response['code']];
        } else {
            $this->setNotificationId($email, $response);
            $notification_token_id = isset($response['notificationTokenId']) ? $response['notificationTokenId'] : false;

            return [
                'success' => true,
                'show_popup' => true,
                'html' => $this->fetchPopupConfirmation($response),
                'notificationTokenId' => $notification_token_id,
                'userUsedPushNotifications' => isset($response['userUsedPushNotifications']) ? $response['userUsedPushNotifications'] : '',
            ];
        }
    }

    public function fetchPopupConfirmation($response)
    {
        $this->context->smarty->assign([
            'module_path' => $this->module->getPathUri(),
        ]);
        if (isset($response['userUsedPushNotifications']) && (bool)$response['userUsedPushNotifications'] === true) {
            return $this->context->smarty->fetch(dirname(__FILE__) . '/../../views/templates/front/confirmation_form.tpl');
        } else {
            $this->context->smarty->assign([
                'code' => $response['code'],
                'data' => $response['data'],
                'message' => $response['message'],
            ]);

            return $this->context->smarty->fetch(dirname(__FILE__) . '/../../views/templates/front/code_form.tpl');
        }
    }

    public function checkAjaxToken()
    {
        $token = Tools::getValue('ajax_token');
        if ($token != $this->module->getAjaxToken()) {
            ['error' => 'Invalid token'];
        }
    }

    public function checkCodeAjaxAction()
    {
        $code = Tools::getValue('code');
        $email = Tools::getValue('email');
        $notification_token_id = Tools::getValue('notificationTokenId');
        $response = $this->api->apiCall(['email' => $email, 'code' => $code, 'notificationTokenId' => $notification_token_id], 'POST', 'checkout/submitCheckoutCodeCommerce');

        $is_code_valid = isset($response['isCodeValid']) ? $response['isCodeValid'] : false;
        if ($is_code_valid) {
            $this->simplyin_session->set('simplyin_auth_token', $response['authToken']);
            $this->simplyin_session->set('simplyin_refresh_token', $response['refreshToken']);
            unset($response['authToken']);
            unset($response['refreshToken']);
            $response['success'] = true;
            $response['login_result'] = $this->loginCustomerByEmail($email, $response);
            $response['html'] = $this->renderDeliveryForm($response);

            return $response;
        } else {
            return $response;
        }
    }

    public function checkPushNotificationAjaxAction()
    {
        $email = $this->simplyin_session->get('simplyin_email');
        $notification_token_id = $this->simplyin_session->get('simplyin_notifiaction_token_id');
        $response = $this->api->apiCall(['email' => $email, 'notificationTokenId' => $notification_token_id], 'POST', 'checkout/checkIfSubmitEmailPushNotificationWasConfirmedCommerce');
        $is_ok = isset($response['ok']) ? $response['ok'] : false;
        if ($is_ok) {
            unset($response['ok']);
            unset($response['rejected']);
            unset($response['request']);
            $this->simplyin_session->set('simplyin_auth_token', $response['authToken']);
            $this->simplyin_session->set('simplyin_refresh_token', $response['refreshToken']);
            $response['login_result'] = $this->loginCustomerByEmail($email, $response);
            $response['success'] = true;
            $response['html'] = $this->renderDeliveryForm($response);

            return $response;
        } else {
            return $response;
        }
    }

    public function userDataCommerceAjaxAction()
    {
        $auth_token = $this->getAuthToken();

        return $this->api->apiCall(false, 'GET', 'userDataCommerce', $auth_token);
    }

    public function logoutAjaxAction()
    {
        $this->module->logout();

        return ['success' => true];
    }

    public function renderAjaxForm($type)
    {
        $result = [];
        $this->context->smarty->assign([
            'module_path' => $this->module->getPathUri(),
            'form_type' => $type,
            'form_action' => $this->context->link->getModuleLink('simplyin', 'ajax', ['action' => 'submitForm']),
        ]);
        if ($type == 'delivery') {
            $result['html'] = $this->renderDeliveryForm();
        } else {
            $result['html'] = $this->context->smarty->fetch(dirname(__FILE__) . '/../../views/templates/front/' . $type . '_form.tpl');
        }
        $result['success'] = true;

        return $result;
    }

    public function renderDeliveryForm($data = false)
    {
        if (!isset($data['data']) && isset($data['userData'])) {
            $data['data'] = $data['userData'];
        }
        if ($data === false || $data === null) {
            $data = $this->userDataCommerceAjaxAction();
        }
        if (isset($data['code']) && $data['code'] == 'UNAUTHORIZED') {
            if (Tools::isSubmit('email')) {
                exit(json_encode(['error' => 'Session expired']));
            }
        }

        if (isset($data['code']) && $data['code'] == 'INTERNAL_SERVER_ERROR') {
            exit(json_encode(['error' => 'Session expired']));
        }
        $carriers_cost = $this->module->getCartCarrierCost();
        $parcel_lockers = $this->getParcelLockers($data, $this->getSelectedParcel());
        $parcel_lockers_min_price = 0;
        $prices = [];
        if (!empty($parcel_lockers)) {
            foreach ($parcel_lockers as $locker) {
                if (isset($carriers_cost[$locker['providerName']])) {
                    $price = (float)str_replace(',', '.', $carriers_cost[$locker['providerName']]);
                    $prices[$price] = $carriers_cost[$locker['providerName']];
                }
            }
        }
        if (!empty($prices)) {
            // value by min key
            $parcel_lockers_min_price = $prices[min(array_keys($prices))];
        }

        $this->context->smarty->assign([
            'module_path' => $this->module->getPathUri(),
            'carriers_cost' => $carriers_cost,
            '_id' => $data['data']['_id'],
            'name' => $data['data']['name'],
            'suername' => $data['data']['surname'],
            'billing_addresses' => $this->parseAddress($data['data']['billingAddresses'], $this->getSelectedBillingAddress()),
            'shipping_addresses' => $this->parseAddress($data['data']['shippingAddresses'], $this->getSelectedShippingAddress()),
            'parcel_lockers' => $parcel_lockers,
            'parcel_lockers_min_price' => $parcel_lockers_min_price,
            'service_type' => $this->getServiceType(),
            'delivery_address' => $this->getSelectedShippingAddress(),
            'billing_address' => $this->getSelectedBillingAddress(),
            'selected_parcel' => $this->getSelectedParcel(),
        ])
        ->assign([
            'form_action' => $this->context->link->getModuleLink('simplyin', 'ajax', ['action' => 'submitForm']),
            'form_type' => 'delivery',
        ]);

        return $this->context->smarty->fetch(dirname(__FILE__) . '/../../views/templates/front/delivery_form.tpl');
    }

    public function getParcelLockers($data, $selected = false)
    {
        $available_carriers = $this->module->getAvailableCarriers();
        if (empty($available_carriers)) {
            return [];
        }
        $parce_lockers = $data['data']['parcelLockers'];
        $sort_by = json_decode(Configuration::get('SIMPLYIN_LOGISTICSPARTNERSORDER'), true);
        if ($sort_by === null) {
            $sort_by = [];
        }

        $sort_by = array_map('strtolower', $sort_by);

        if (!empty($sort_by)) {
            usort($parce_lockers, function ($a, $b) use ($sort_by) {
                $a_index = array_search(strtolower($a['providerName']), $sort_by);
                $b_index = array_search(strtolower($b['providerName']), $sort_by);

                if ($a_index === false) $a_index = PHP_INT_MAX;
                if ($b_index === false) $b_index = PHP_INT_MAX;

                if ($a_index !== $b_index) {
                    return $a_index - $b_index;
                }

                return strcasecmp($a['providerName'], $b['providerName']);
            });
        }

        foreach ($parce_lockers as $key => $parcel) {
            if (!isset($parcel['providerName']) || !in_array($parcel['providerName'], $available_carriers)) {
                unset($parce_lockers[$key]);
                continue;
            }
            if (empty($parcel['addressName'])) {
                $parce_lockers[$key]['addressName'] = $parcel['lockerId'];
            }
            if ($selected === $parcel['_id'] || $selected === false && $parcel['service_type'] == 'parcel_machine') {
                $parce_lockers[$key]['selected'] = true;
                $selected = $parcel['_id'];
            } else {
                $parce_lockers[$key]['selected'] = false;
            }
        }

        return $parce_lockers;
    }

    public function parseAddress($addresses, $selected)
    {
        foreach ($addresses as $key => $address) {
            $addresses[$key]['selected'] = false;
        }

        foreach ($addresses as $key => $address) {
            if ($address['_id'] == $selected || $selected === false) {
                $addresses[$key]['selected'] = true;
                break;
            }
        }

        return $addresses;
    }

    public function getSelectedShippingAddress()
    {
        $shipping = $this->simplyin_session->get('simplyin_shipping', '');
        if (isset($shipping) && !empty($shipping)) {
            return $shipping;
        } else {
            return false;
        }
    }

    public function getSelectedParcel()
    {
        $parcel = $this->simplyin_session->get('simplyin_parcel', '');
        if (isset($parcel) && !empty($parcel)) {
            return $parcel;
        } else {
            return false;
        }
    }

    public function getSelectedBillingAddress()
    {
        $billing = $this->simplyin_session->get('simplyin_billing', '');

        if (isset($billing) && !empty($billing)) {
            return $billing;
        } else {
            return false;
        }
    }

    public function getServiceType()
    {
        $service_type = $this->simplyin_session->get('service_type', '');
        if (isset($service_type) && !empty($service_type)) {
            return $service_type;
        } else {
            return false;
        }
    }

    public function deleteAddressAjaxAction()
    {
        $data = $this->userDataCommerceAjaxAction();
        $data = ['userData' => $data['data']];
        $id = Tools::getValue('id');
        $auth_token = $this->getAuthToken();
        $delete_from = '';
        $address_type = Tools::getValue('address_type');
        if ($address_type == 'billingAddresses') {
            foreach ($data['userData']['billingAddresses'] as $key => $item) {
                if ($item['_id'] == $id) {
                    unset($data['userData']['billingAddresses'][$key]);
                    $data['userData']['billingAddresses'] = array_values($data['userData']['billingAddresses']);
                    $delete_from = 'billingAddresses';
                    break;
                }
            }
        } else if ($address_type == 'shippingAddresses') {
            foreach ($data['userData']['shippingAddresses'] as $key => $item) {
                if ($item['_id'] == $id) {
                    unset($data['userData']['shippingAddresses'][$key]);
                    $data['userData']['shippingAddresses'] = array_values($data['userData']['shippingAddresses']);
                    $delete_from = 'shippingAddresses';
                    break;
                }
            }
        } else {
            foreach ($data['userData']['parcelLockers'] as $key => $item) {
                if ($item['_id'] == $id) {
                    unset($data['userData']['parcelLockers'][$key]);
                    $data['userData']['parcelLockers'] = array_values($data['userData']['parcelLockers']);
                    $delete_from = 'parcelLockers';
                    break;
                }
            }
        }

        $result = $this->api->apiCall($data, 'PATCH', 'userData', $auth_token);

        return [
            $delete_from => $result['data'][$delete_from],
            'success' => true,
        ];
    }

    public function getPhoneNumber($data)
    {
        $data = isset($data['data']) ? $data['data'] : $data['userData'];

        return $data['phoneNumber'];
    }

    public function findAddress($data, $id, $type = '')
    {
        $data = isset($data['data']) ? $data['data'] : $data['userData'];
        $address_type = '';
        $address = false;
        if ($type == 'billingAddresses') {
            foreach ($data['billingAddresses'] as $item) {
                if ($item['_id'] == $id) {
                    $address = $item;
                    $address_type = 'billingAddresses';
                }
            }
        } else if ($type == 'shippingAddresses') {
            foreach ($data['shippingAddresses'] as $item) {
                if ($item['_id'] == $id) {
                    $address = $item;
                    $address_type = 'shippingAddresses';
                }
            }
        } else {
            foreach ($data['parcelLockers'] as $item) {
                if ($item['_id'] == $id) {
                    $address = $item;
                    $address_type = 'parcelLockers';
                }
            }
        }


        return [
            'address' => $address,
            'address_type' => $address_type,
        ];
    }

    public function getDefaultCountryIso()
    {
        $default_country = Configuration::get('PS_COUNTRY_DEFAULT');
        $country = new Country($default_country);

        return $country->iso_code;
    }

    public function editAddressAjaxAction()
    {
        $id = Tools::getValue('id');
        $data = $this->userDataCommerceAjaxAction();
        $address_type = Tools::getValue('address_type');
        $address_data = $this->findAddress($data, $id, $address_type);
        $this->context->smarty->assign([
            'default_country_iso' => $this->getDefaultCountryIso(),
            'address' => $address_data['address'],
            'address_type' => $address_data['address_type'],
            'countries' => Country::getCountries($this->context->language->id),
            'module_path' => $this->module->getPathUri(),
        ]);
        $result = [
            'success' => true,
            'html' => $this->context->smarty->fetch(dirname(__FILE__) . '/../../views/templates/front/edit_address_form.tpl'),
        ];

        return $result;
    }

    public function createEmptyAddress($address_type)
    {
        return [
            'address_type' => $address_type,
            'address' => [
                '_id' => '',
                'name' => '',
                'surname' => '',
                'street' => '',
                'postalCode' => '',
                'city' => '',
                'companyName' => '',
                'appartmentNumber' => '',
                'addressName' => '',
                'taxId' => '',
                'country' => $this->getDefaultCountryIso(),
            ],
        ];
    }

    public function addAddressAjaxAction()
    {
        $address_type = Tools::getValue('address_type');
        if ($address_type != 'parcel') {
            $data = $this->userDataCommerceAjaxAction();
            $address_data = $this->createEmptyAddress($address_type);
            $this->context->smarty->assign([
                'default_country_iso' => $this->getDefaultCountryIso(),
                'address' => $address_data['address'],
                'address_type' => $address_data['address_type'],
                'countries' => Country::getCountries($this->context->language->id),
                'module_path' => $this->module->getPathUri(),
            ]);
            $result = [
                'success' => true,
                'html' => $this->context->smarty->fetch(dirname(__FILE__) . '/../../views/templates/front/edit_address_form.tpl'),
            ];

            return $result;
        } else {
            $this->context->smarty->assign([
                'module_path' => $this->module->getPathUri(),
            ]);
            $result = [
                'success' => true,
                'html' => $this->context->smarty->fetch(dirname(__FILE__) . '/../../views/templates/front/autocomplete_form.tpl'),
            ];

            return $result;
        }
    }

    public function validateAddress($address)
    {
        $errors = [];
        if ($address['address_type'] == 'parcelLockers') {
            return true;
        } else {
            $firstname = $address['name'];
            $lastname = $address['surname'];
            $street = $address['street'];
            $postal_code = $address['postalCode'];
            $city = $address['city'];
            $companyName = $address['companyName'];
            $addressName = $address['addressName'];
            $taxId = $address['taxId'];

            if (!Validate::isCustomerName($firstname)) {
                $errors['name'] = $this->l('Invalid name');
            }
            if (!Validate::isGenericName($taxId)) {
                $errors['taxId'] = $this->l('Invalid tax ID');
            }
            if (!Validate::isCustomerName($lastname)) {
                $errors['surname'] = $this->l('Invalid surname');
            }
            if (!Validate::isGenericName($companyName)) {
                $errors['companyName'] = $this->l('Invalid company name');
            }
            if (!Validate::isGenericName($addressName)) {
                $errors['addressName'] = $this->l('Invalid address name');
            }

            if (!Validate::isAddress($street)) {
                $errors['street'] = $this->l('Invalid street');
            }
            if (!Validate::isCityName($city)) {
                $errors['city'] = $this->l('Invalid city');
            }
            if (!Validate::isPostCode($postal_code)) {
                $errors['postalCode'] = $this->l('Invalid postal code');
            }
            $id_country = Country::getByIso($address['country']);
            $country = new Country($id_country);
            if ($country->id) {
                if (!$country->checkZipCode($postal_code)) {
                    $errors['postalCode'] = sprintf($this->l('Invalid postal code: %s'), $country->zip_code_format);
                }
            }
            if (empty($errors)) {
                return true;
            } else {
                return $errors;
            }
        }
    }

    public function saveExistingAddress($address, $id)
    {
        $is_valid = $this->validateAddress($address);
        $auth_token = $this->getAuthToken();
        if ($is_valid === true) {
            $data = $this->userDataCommerceAjaxAction();
            $data = ['userData' => $data['data']];
            $address_type = $address['address_type'];
            // if ($address_type == 'billingAddresses')
            {
                foreach ($data['userData']['billingAddresses'] as $key => $item) {
                    if ($item['_id'] == $id) {
                        foreach ($address as $address_key => $value) {
                            if (isset($data['userData']['billingAddresses'][$key][$address_key])) {
                                if ($address_key == 'taxId' && $address_type == 'shippingAddresses') {
                                    continue;
                                }
                                $data['userData']['billingAddresses'][$key][$address_key] = $value;
                            }
                        }
                        break;
                    }
                }
            }
            // else
            {
                foreach ($data['userData']['shippingAddresses'] as $key => $item) {
                    if ($item['_id'] == $id) {
                        foreach ($address as $address_key => $value) {
                            if ($address_key == 'taxId' && $address_type == 'shippingAddresses') {
                                continue;
                            }
                            if (isset($data['userData']['shippingAddresses'][$key][$address_key])) {
                                $data['userData']['shippingAddresses'][$key][$address_key] = $value;
                            }
                        }
                        break;
                    }
                }
            }

            $result = $this->api->apiCall($data, 'PATCH', 'userData', $auth_token);
            $address_data = $this->findAddress($result, $id, $address_type);

            return [
                'address_type' => $address_type,
                'address' => $address_data['address'],
                'is_valid' => $is_valid,
                'success' => true,
            ];
        } else {
            return [
                'is_valid' => $is_valid,
                'success' => true,
            ];
        }
    }

    public function createNewAddress($address)
    {
        $is_valid = $this->validateAddress($address);
        $auth_token = $this->getAuthToken();
        if ($is_valid === true) {
            $data = $this->userDataCommerceAjaxAction();
            $data = ['userData' => $data['data']];
            $address_type = $address['address_type'];
            if ($address_type == 'billingAddresses') {
                $data['userData']['billingAddresses'][] = $address;
            } elseif ($address_type == 'shippingAddresses') {
                $data['userData']['shippingAddresses'][] = $address;
            } elseif ($address_type == 'parcelLockers') {
                unset($address['address_type']);
                $data['userData']['parcelLockers'][] = $address;
            }
            $result = $this->api->apiCall($data, 'PATCH', 'userData', $auth_token);
            $address = $result['data'][$address_type][sizeof($result['data'][$address_type]) - 1];
            $result = $this->userDataCommerceAjaxAction();

            return [
                'result' => $result,
                'add_address' => $address = $result['data'][$address_type][sizeof($result['data'][$address_type]) - 1],
                'address_type' => $address_type,
                'html' => $this->renderDeliveryForm($result),
                'is_valid' => $is_valid,
                'success' => true,
            ];
        } else {
            return [
                'is_valid' => $is_valid,
                'success' => true,
            ];
        }
    }

    public function submitAddressAjaxAction()
    {
        $parcel = false;
        $address = false;
        if (Tools::isSubmit('address')) {
            $address = Tools::getValue('address');
            $parcel = false;
        } else {
            $address = Tools::getValue('parcel');
            $parcel = true;
        }
        if (empty($address)) {
            return ['error' => 'Invalid address'];
        }
        $id = $address['_id'];
        if (empty($id) || $parcel) {
            return $this->createNewAddress($address);
        } else {
            return $this->saveExistingAddress($address, $id);
        }
    }

    public function searchAddressAjaxAction()
    {
        $auth_token = $this->getAuthToken();
        $address = Tools::getValue('address');
        $result = $this->api->apiCall(['searchAddressBy' => $address], 'POST', 'addresses/find', $auth_token);
        $result['success'] = true;

        return $result;
    }

    public function getAuthToken()
    {
        return $this->simplyin_session->get('simplyin_auth_token', false);
    }

    public function getClosestParcelLockersAjaxAction()
    {
        $available_carriers = $this->module->getAvailableCarriers();
        if (empty($available_carriers)) {
            return [];
        }
        $result['available_carriers'] = $available_carriers;
        $auth_token = $this->getAuthToken();
        $coordinates = Tools::getValue('coordinates');
        $params = [
            'numberOfItemsToFind',
            'searchRadiusInMeters',
            'coordinates' => [
                'lng' => (float) $coordinates['lng'],
                'lat' => (float) $coordinates['lat'],
            ],
            'acceptedParcelLockerProviders' => $available_carriers,
        ];
        $result = $this->api->apiCall($params, 'POST', 'parcelLockers/getClosest', $auth_token);
        $sort_by = json_decode(Configuration::get('SIMPLYIN_LOGISTICSPARTNERSORDER'), true);
        $sort_by = array_map('strtolower', $sort_by);

        if (!empty($sort_by)) {
            usort($result['data'], function ($a, $b) use ($sort_by) {
                $a_index = array_search(strtolower($a['info']['provider']['name']), $sort_by);
                $b_index = array_search(strtolower($b['info']['provider']['name']), $sort_by);

                if ($a_index === false) $a_index = PHP_INT_MAX;
                if ($b_index === false) $b_index = PHP_INT_MAX;

                if ($a_index !== $b_index) {
                    return $a_index - $b_index;
                }

                return strcasecmp($a['info']['provider']['name'], $b['info']['provider']['name']);
            });
        }
        return [
            'success' => true,
            'data' => $result,
        ];
    }

    public function loginCustomerByEmail($email, $response_data)
    {
        if (!isset($response_data['data']) && isset($response_data['userData'])) {
            $response_data['data'] = $response_data['userData'];
        }
        $customer = new Customer();
        $customer->getByEmail($response_data['data']['email']);
        $result = [];
        if ($customer->id) {
            if (!$customer->is_guest) {
                Hook::exec('actionBeforeAuthentication');

                if (class_exists('CompareProduct')) {
                    $this->context->cookie->id_compare = isset($this->context->cookie->id_compare) ? $this->context->cookie->id_compare : CompareProduct::getIdCompareByIdCustomer($customer->id);
                }
                $this->context->cookie->id_customer = (int) ($customer->id);
                $this->context->updateCustomer($customer);
            }
            $result['is_guest'] = false;
        } else {
            $customer->email = $email;
            $customer->passwd = Tools::encrypt(Tools::passwdGen());
            $customer->is_guest = 1;
            $customer->firstname = $response_data['data']['name'];
            $customer->lastname = $response_data['data']['surname'];
            $customer->add();
            $this->context->cookie->id_customer = (int) ($customer->id);
            $this->context->updateCustomer($customer);
            $result['is_guest'] = true;
            $result['firstname'] = $response_data['data']['name'];
            $result['lastname'] = $response_data['data']['surname'];
        }

        return $result;
    }

    public function deleteOtherGuestAddresses($id_customer, $id_address_invoice, $id_address_delivery)
    {
        $sql = '
            SELECT      id_address
            FROM        `' . _DB_PREFIX_ . 'address`
            WHERE       id_customer = ' . (int) $id_customer . '
                AND id_address != ' . (int) $id_address_invoice . '
                AND id_address != ' . (int) $id_address_delivery . '
        ';
        $result = Db::getInstance()->executeS($sql);
        if (!empty($result)) {
            foreach ($result as $item) {
                $address = new Address($item['id_address']);
                $address->delete();
            }
        }
    }

    public function selectAddressAjaxAction()
    {
        $opc = $this->module->getOPC();
        $invoice_address_id = Tools::getValue('invoice_address');
        $delivery_address_id = Tools::getValue('delivery_address');
        if ($delivery_address_id == '') {
            $delivery_address_id = $invoice_address_id;
        }

        $id_selected_point = Tools::getValue('selected_point');
        if (!empty($id_selected_point)) {
            $this->selectDeliveryPoint($id_selected_point);
        } else {
            $this->clearDeliveryPoint();
        }
        $service_type = Tools::getValue('service_type');

        $this->simplyin_session->set('simplyin_billing', $invoice_address_id);
        $this->simplyin_session->set('simplyin_shipping', $delivery_address_id);
        $this->simplyin_session->set('simplyin_parcel', $id_selected_point);
        $this->simplyin_session->set('service_type', $service_type);

        $data = $this->userDataCommerceAjaxAction();
        $invoice_address = $this->findAddress($data, $invoice_address_id , 'billingAddresses');
        $ps_id_address_invoice = $this->checkIfAddressExists($this->context->customer->id, $invoice_address['address']);
        $phone = $this->getPhoneNumber($data);
        if (!$ps_id_address_invoice) {
            $ps_id_address_invoice = $this->createNewAddressPS($this->context->customer->id, $invoice_address['address'], $phone);
        }
        if ($invoice_address_id != $delivery_address_id) {
            $delivery_address = $this->findAddress($data, $delivery_address_id, 'shippingAddresses');
        } else {
            $delivery_address = $invoice_address;
        }

        $ps_id_address_delivery = $this->checkIfAddressExists($this->context->customer->id, $delivery_address['address']);
        if (!$ps_id_address_delivery) {
            $ps_id_address_delivery = $this->createNewAddressPS($this->context->customer->id, $delivery_address['address'], $phone);
        }

        if ($this->context->customer->isGuest()) {
            $this->deleteOtherGuestAddresses($this->context->customer->id, $ps_id_address_invoice, $ps_id_address_delivery);
        }
        $this->simplyin_session->set('simplyin_id_address_delivery', $ps_id_address_delivery);
        $this->simplyin_session->set('simplyin_id_address_invoice', $ps_id_address_invoice);

        $this->context->cart->id_address_delivery = (int) $ps_id_address_delivery;
        $this->context->cart->id_address_invoice = (int) $ps_id_address_invoice;
        $this->context->cart->delivery_option = $this->findDeliveryOption($this->context->cart);
        $this->context->cart->update();

        $this->updateDeliveryaddress($ps_id_address_delivery, $this->context->cart->id);

        if (!$this->context->customer->isGuest() || $opc == 'thecheckout') {
            $checkout_session_data = $this->module->getCheckoutSessionData();
            $this->module->upgradeSessionData($checkout_session_data, false);
        }

        return [
            'success' => true,
            'warnings' => [],
            'redirect' => $this->needRedirect(),
        ];
    }

    public function needRedirect()
    {
        $opc = $this->module->getOPC();
        if ($opc == 'thecheckout') {
            return true;
        } else {
            return !$this->context->customer->isGuest();
        }
    }

    private function updateDeliveryaddress($id_address_delivery, $id_cart)
    {
        $sql = '
            UPDATE      `' . _DB_PREFIX_ . 'cart_product`
            SET         id_address_delivery = ' . (int) $id_address_delivery . '
            WHERE       id_cart = ' . (int) $id_cart;

        Db::getInstance()->execute($sql);
        $sql = '
            UPDATE      `' . _DB_PREFIX_ . 'customization`
            SET         id_address_delivery = ' . (int) $id_address_delivery . '
            WHERE       id_cart = ' . (int) $id_cart;
        Db::getInstance()->execute($sql);
    }

    private function checkIfAddressExists($id_customer, $address_data)
    {
        $sql = '
        SELECT  id_address, alias, firstname, lastname, address1, address2, postcode, city,
                id_country, vat_number, company, phone
        FROM ' . _DB_PREFIX_ . 'address
        WHERE id_customer = ' . (int) $id_customer . '
            AND deleted = 0
        ';

        $result = Db::getInstance()->executeS($sql);
        if (!isset($address_data['taxId'])) {
            $address_data['taxId'] = '';
        }

        foreach ($result as $item) {
            $exploded = explode(' ', $item['address1']);
            if (!isset($exploded[1])) {
                $exploded[1] = '';
            }
            if (
                strcasecmp($item['alias'], $address_data['addressName']) === 0
                && strcasecmp($item['firstname'], $address_data['name']) === 0
                && strcasecmp($item['lastname'], $address_data['surname']) === 0
                && strcasecmp($exploded[0], $address_data['street']) === 0
                && strcasecmp($exploded[1], $address_data['appartmentNumber']) === 0
                && strcasecmp($item['postcode'], $address_data['postalCode']) === 0
                && strcasecmp($item['city'], $address_data['city']) === 0 && $item['id_country'] == $address_data['country']
                && strcasecmp($item['vat_number'], $address_data['taxId']) === 0
                && strcasecmp($item['company'], $address_data['companyName']) === 0
            ) {
                return $item['id_address'];
            }
        }

        return false;
    }

    private function createNewAddressPS($id_customer, $address_data, $phone)
    {
        $id_country = Country::getByIso('PL');
        $address_data['country'] = $id_country;
        if (($id_address = $this->checkIfAddressExists($id_customer, $address_data))) {
            return $id_address;
        }

        $address = new Address();
        $address->id_customer = $id_customer;
        $address->alias = substr($address_data['addressName'], 0, 32);
        $address->firstname = substr($address_data['name'], 0, 255);
        $address->lastname = substr($address_data['surname'], 0, 255);
        $ap_number = trim($address_data['appartmentNumber']);
        if (!empty($ap_number)) {
            $ap_number = ' ' . $ap_number;
        } else {
            $ap_number = '';
        }

        $address->address1 = substr($address_data['street'].$ap_number, 0, 128);
        $address->postcode = $address_data['postalCode'];
        $address->city = substr($address_data['city'], 0, 64);
        $address->id_country = $address_data['country'];
        $address->vat_number = $address_data['taxId'] ?? '';
        $address->company = $address_data['companyName'];

        $address->phone = $phone;

        if ($address->save()) {
            return $address->id; // Address added successfully
        } else {
            return false; // Address couldn't be added
        }
    }

    public function selectDeliveryPoint($point_id)
    {
        $id_cart = $this->context->cart->id;
        $data = $this->findAddress($this->userDataCommerceAjaxAction(), $point_id);
        $point = SimplyInPoint::getByIdCart($id_cart);

        $point->providerId = $data['address']['providerId'];
        $point->service_type = $data['address']['service_type'];
        $point->addressName = $data['address']['addressName'];
        $point->address = $data['address']['address'];
        $point->label = $data['address']['providerName'];
        $point->lockerId = $data['address']['lockerId'];
        $point->save();

        if ($point->label == 'inpost') {
            $this->setInpostPoint($point);
        } elseif ($point->label == 'dhl') {
            $this->setDhlPoint($point);
        } elseif ($point->label == 'dpd') {
            $this->setDpdPoint($point);
        } elseif ($point->label == 'orlen') {
            $this->setOrlenPoint($point);
        } elseif ($point->label == 'fedex') {
            $this->setFedexPoint($point);
        } elseif ($point->label == 'ups') {
            $this->setUpsPoint($point);
        }
    }

    public function clearDeliveryPoint()
    {
        $id_cart = $this->context->cart->id;
        $point = SimplyInPoint::getByIdCart($id_cart);

        if ($point->id) {
            if ($point->label == 'inpost') {
                $this->clearInpostPoint();
            } elseif ($point->label == 'dhl') {
                $this->clearDhlPoint();
            } elseif ($point->label == 'dpd') {
                $this->clearDpdPoint();
            } elseif ($point->label == 'orlen') {
                $this->clearOrlenPoint();
            } elseif ($point->label == 'fedex') {
                $this->clearFedexPoint();
            } elseif ($point->label == 'ups') {
                $this->clearUpsPoint();
            }
            $point->delete();
        }
        $this->setCarrier(Configuration::get('SIMPLYIN_DEFAULT_CARRIER_ID'));
    }

    public function setUpsPoint($point)
    {
        $this->setApaczkaPoint($point);
    }

    public function clearUpsPoint()
    {
        $this->clearApackaPoint();
    }

    public function setInpostPoint($point)
    {
        $id_cart = $this->context->cart->id;
        $this->setCarrier(Configuration::get('SIMPLYIN_INPOST_CARRIER_ID'));
        $this->context->cart->update();
        if (Module::isInstalled('pminpostpaczkomaty') && Module::isEnabled('pminpostpaczkomaty')) {
            try {
                require_once _PS_MODULE_DIR_ . 'pminpostpaczkomaty/classes/PaczkomatyList.php';
                $paczkomaty_list = PaczkomatyList::getByIdCart($id_cart);
                $paczkomaty_list->machine = pSql($point->lockerId);
                $paczkomaty_list->save();
            } catch (Exception $exp) {
                $this->addWarning($exp->getMessage());
            }
        }
        $this->setApaczkaPoint($point);
        if (Module::isInstalled('pdinpostpaczkomaty') && Module::isEnabled('pdinpostpaczkomaty')) {
            try {
                require_once _PS_MODULE_DIR_ . 'pdinpostpaczkomaty/classes/PaczkomatyList.php';
                $paczkomaty_list = PaczkomatyList::getByIdCart($id_cart);
                $paczkomaty_list->machine = pSql($point->lockerId);
                $paczkomaty_list->save();
            } catch (Exception $exp) {
                $this->addWarning($exp->getMessage());
            }
        }
        if (Module::isInstalled('inpostshipping') && Module::isEnabled('inpostshipping')) {
            try {
                $sql = '
                DELETE FROM `' . _DB_PREFIX_ . 'inpost_cart_choice`
                    WHERE `id_cart` = ' . (int) $id_cart;
                Db::getInstance()->execute($sql);

                $sql = 'INSERT INTO `' . _DB_PREFIX_ . 'inpost_cart_choice` (`id_cart`, `service`, `point`)
                    VALUES (' . (int) $id_cart . ', "inpost_locker_standard", "' . pSql($point->lockerId) . '")';
                Db::getInstance()->execute($sql);
            } catch (Exception $exp) {
                $this->addWarning($exp->getMessage());
            }
        }
    }

    public function clearInpostPoint()
    {
        $id_cart = $this->context->cart->id;

        if (Module::isInstalled('pminpostpaczkomaty') && Module::isEnabled('pminpostpaczkomaty')) {
            try {
                require_once _PS_MODULE_DIR_ . 'pminpostpaczkomaty/classes/PaczkomatyList.php';
                $paczkomaty_list = PaczkomatyList::getByIdCart($id_cart);
                $paczkomaty_list->delete();
            } catch (Exception $exp) {
                $this->addWarning($exp->getMessage());
            }
        }
        if (Module::isInstalled('pdinpostpaczkomaty') && Module::isEnabled('pdinpostpaczkomaty')) {
            try {
                require_once _PS_MODULE_DIR_ . 'pdinpostpaczkomaty/classes/PaczkomatyList.php';
                $paczkomaty_list = PaczkomatyList::getByIdCart($id_cart);
                $paczkomaty_list->delete();
            } catch (Exception $exp) {
                $this->addWarning($exp->getMessage());
            }
        }
        if (Module::isInstalled('inpostshipping') && Module::isEnabled('inpostshipping')) {
            try {
                $sql = '
                    DELETE FROM `' . _DB_PREFIX_ . 'inpost_cart_choice`
                    WHERE `id_cart` = ' . (int) $id_cart;
                Db::getInstance()->execute($sql);
            } catch (Exception $exp) {
                $this->addWarning($exp->getMessage());
            }
        }
        $this->clearApackaPoint();
    }

    public function setDhlPoint($point)
    {
        $id_cart = $this->context->cart->id;
        $this->setCarrier(Configuration::get('SIMPLYIN_DHL_CARRIER_ID'));
        $this->context->cart->update();

        $this->setApaczkaPoint($point);
    }

    public function clearDhlPoint()
    {
        $id_cart = $this->context->cart->id;

        $this->clearApackaPoint();
    }

    public function setDpdPoint($point)
    {
        $id_cart = $this->context->cart->id;
        $this->setCarrier(Configuration::get('SIMPLYIN_DPD_CARRIER_ID'));
        $this->context->cart->update();
        if (Module::isInstalled('dpdshipping')) {
            $sql = '
                DELETE FROM `' . _DB_PREFIX_ . 'dpdshipping_cart_pickup`
                    WHERE `id_cart` = ' . (int) $id_cart;
            Db::getInstance()->execute($sql);

            $sql = 'INSERT INTO `' . _DB_PREFIX_ . 'dpdshipping_cart_pickup` (`id_cart`, `id_shop`, `pudo_code`)
                VALUES (' . (int) $id_cart . ', ' . (int) $this->context->shop->id . ', "' . pSql($point->lockerId) . '")';
            Db::getInstance()->execute($sql);
        }
        $this->setApaczkaPoint($point);
    }

    public function clearDpdPoint()
    {
        $sql = '
            DELETE FROM `' . _DB_PREFIX_ . 'dpdshipping_cart_pickup`
                WHERE `id_cart` = ' . (int) $this->context->cart->id;
        Db::getInstance()->execute($sql);
        $this->clearApackaPoint();
    }

    public function setOrlenPoint($point)
    {
        $id_cart = $this->context->cart->id;
        $this->setCarrier(Configuration::get('SIMPLYIN_ORLEN_CARRIER_ID'));
        $this->context->cart->update();
    }

    public function clearOrlenPoint()
    {
        $this->clearApackaPoint();
    }

    public function setFedexPoint($point)
    {
        $id_cart = $this->context->cart->id;
        $this->setCarrier(Configuration::get('SIMPLYIN_FEDEX_CARRIER_ID'));
        $this->context->cart->update();
    }

    public function clearFedexPoint()
    {

    }

    public function setCarrier($id_carrier)
    {
        $id_carrier = (int) $id_carrier;
        $cart = $this->context->cart;

        if (!$cart || !$id_carrier) {
            return;
        }

        $id_address_delivery = (int) $cart->id_address_delivery;
        if (!$id_address_delivery && $cart->id_customer) {
            $id_address_delivery = (int) Address::getFirstCustomerAddressId($cart->id_customer);
        }
        if (!$id_address_delivery) {
            return;
        }

        $deliveryOption = [
            $id_address_delivery => $id_carrier . ',',
        ];

        $cart->setDeliveryOption($deliveryOption);
        $cart->id_carrier = $id_carrier;
        $cart->update();

        if (Module::isEnabled('thecheckout')) {
            $cookie = $this->context->cookie;
            $opc_form_radios = [];

            if (!empty($cookie->opc_form_radios)) {
                $decoded = json_decode($cookie->opc_form_radios, true);
                if (is_array($decoded)) {
                    $opc_form_radios = $decoded;
                }
            }

            $opc_form_radios['delivery_option'] = $deliveryOption;
            $cookie->opc_form_radios = json_encode($opc_form_radios);

            if (class_exists('\PrestaShop\PrestaShop\Adapter\Adapter_ServiceLocator')) {
                $checkoutSession = \PrestaShop\PrestaShop\Adapter\Adapter_ServiceLocator::get('prestashop.core.checkout.checkout_session');
                if ($checkoutSession) {
                    $checkoutSession->setDeliveryOption($deliveryOption);
                }
            }
        }

        CartRule::autoRemoveFromCart($this->context);
        CartRule::autoAddToCart($this->context);
    }



    public function addWarning($msg)
    {
        $this->warnings[] = $msg;
    }

    public function findDeliveryOption($cart)
    {
        $return = [
            $cart->id_address_delivery => $cart->id_carrier . ',',
        ];

        return json_encode($return);
    }

    public function createOrderWithoutaccount($order)
    {
        $id_order = $order->id;
        $orderData = SimplyInOrderSync::getOrderData($id_order);
        $orderData['newOrderData']['placedDuringAccountCreation'] = false;
        unset($orderData['newAccountData']);
        $result = $this->api->apiCall($orderData, 'POST', 'checkout/createOrderWithoutaccount', $this->getAuthToken());

        return $result;
    }

    public function createOrderAndAccount($order)
    {
        $id_order = $order->id;
        $customer = new Customer($order->id_customer);
        SimplyInEmailCache::clearCacheByEmail($customer->email);
        $orderData = SimplyInOrderSync::getOrderData($id_order);
        $orderData['newOrderData']['placedDuringAccountCreation'] = true;
        $result = $this->api->apiCall($orderData, 'POST', 'checkout/createOrderAndAccount');

        return $result;
    }

    public function placeOrderAjaxAction()
    {
        $id_cart = (int) Tools::getValue('id_cart');
        $order_token = Tools::getValue('order_token');
        $order_token_module = $this->module->getOrderToken($id_cart);
        if ($order_token != $order_token_module) {
            return [
                'success' => false,
                'error' => $this->l('Invalid order token'),
            ];
        } else {
            $order = Order::getByCartId($id_cart);
            $result = [];
            $new_account_phone = $this->simplyin_session->get('new_account_phone', false);
            $auth_token = $this->getAuthToken();
            if (isset($auth_token) && !empty($auth_token)) {
                $result = $this->createOrderWithoutaccount($order);
                $this->simplyin_session->set('simplyin_auth_token', $auth_token);
            } elseif (isset($new_account_phone) && !empty($new_account_phone)) {
                $result = $this->createOrderAndAccount($order);
                $this->simplyin_session->unset('new_account_phone', '');
            } else {
                return [
                    'success' => false,
                    'error' => $this->l('Invalid order'),
                ];
            }
            $this->simplyin_session->unset('simplyin_id_address_delivery', '');
            $this->simplyin_session->unset('simplyin_id_address_invoice', '');
            if ($result) {
                if (isset($result['createdOrder']['_id'])) {
                    $object = SimplyInOrder::getByIdOrder($order->id);
                    $object->simplyin_id = $result['createdOrder']['_id'];
                    $object->save();
                }
            }
            $this->simplyin_session->clearSession();

            return [
                'success' => true,
                'result' => $result,
            ];
        }
    }

    public function loadCountries()
    {
        $countries_json = file_get_contents(dirname(__FILE__) . '/../../views/json/countries.json');
        $countries = json_decode($countries_json, true);

        return $countries;
    }

    public function getNewAccountFormAjaxAction()
    {
        $this->context->smarty->assign([
            'module_path' => $this->module->getPathUri(),
            'countries' => $this->loadCountries(),
        ]);
        $docs = Configuration::get('SIMPLYIN_DOCS_INFO');
        $docs = json_decode($docs, true);
        if (empty($docs)) {
            $docs = [
                'tc-b2b-pl' => 'https://www.simply.in/terms-b2b',
                'tc-b2c-pl' => 'https://www.simply.in/terms-b2c',
                'tc-b2c-en' => 'https://www.simply.in/en/terms-b2c',
                'tc-b2b-en' => 'https://www.simply.in/en/terms-b2b',
                'gdpr-en' => 'https://www.simply.in/en/privacy-policy',
                'gdpr-pl' => 'https://www.simply.in/privacy-polic',
            ];
        }
        $docs_language = 'en';
        if ($this->context->language->iso_code == 'pl') {
            $docs_language = 'pl';
        }
        $prefix = '+48';
        $checked = (int) Configuration::get('SIMPLYIN_SAVE_CHECKBOX');
        $phone = $this->module->getNewAccountPhone();
        if ($phone !== false) {
            $phone_exploded = explode(' ', $phone);
            if (count($phone_exploded) > 1) {
                $prefix = $phone_exploded[0];
                $phone = $phone_exploded[1];
                $checked = true;
            }
        } else {
            $id_address_delivery = $this->context->cart->id_address_delivery;
            $address = new Address($id_address_delivery);
            $phone = $address->phone;
        }
        $this->context->smarty->assign([
            'terms_link' => $docs['tc-b2b-' . $docs_language],
            'privacy_link' => $docs['gdpr-' . $docs_language],
            'selected_prefix' => $prefix,
            'phone' => $phone,
            'checked' => $checked,
        ]);

        return [
            'success' => true,
            'html' => $this->context->smarty->fetch(dirname(__FILE__) . '/../../views/templates/front/create_account.tpl'),
        ];
    }

    public function storeNewAccountDetailsAjaxAction()
    {
        $phone = Tools::getValue('phone');
        if ($phone) {
            $this->simplyin_session->set('new_account_phone', $phone);
        } else {
            $this->simplyin_session->set('new_account_phone', '');
        }

        return [
            'success' => true,
        ];
    }

    public function clearApackaPoint()
    {
        $id_cart = $this->context->cart->id;
        if (Module::isInstalled('apaczka') && Module::isEnabled('apaczka')) {
            try {
                $sql = '
                    UPDATE `' . _DB_PREFIX_ . 'cart`
                    SET
                        apaczka_supplier = "",
                        apaczka_point = ""
                    WHERE `id_cart` = ' . (int) $id_cart;
                Db::getInstance()->execute($sql);
            } catch (Exception $exp) {
                $this->addWarning($exp->getMessage());
            }
        }
    }

    public function setApaczkaPoint($point)
    {
        if (Module::isInstalled('apaczka') && Module::isEnabled('apaczka')) {
            $id_cart = $this->context->cart->id;
            try {
                $sql = '
                    UPDATE `' . _DB_PREFIX_ . 'cart`
                    SET
                        apaczka_supplier = "'.$point->label.'",
                        apaczka_point = "' . pSql($point->lockerId) . '"
                    WHERE `id_cart` = ' . (int) $id_cart;
                Db::getInstance()->execute($sql);
            } catch (Exception $exp) {
                $this->addWarning($exp->getMessage());
            }
        }
    }
}
