<?php
/**
 * Copyright 2024-2025 Simply.IN Sp. z o.o.
 *
 * NOTICE OF LICENSE
 *
 * Licensed under the EUPL-1.2 or later.
 * You may not use this work except in compliance with the Licence.
 *
 * Copy of the Licence is available at:
 * https://joinup.ec.europa.eu/software/page/eupl
 * It is bundled with this package in the file LICENSE.txt
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the Licence is distributed on an as is basis,
 * without warranties or conditions of any kind, either express or implied.
 * Check the Licence for the specific language governing permissions
 * and limitations under the License.
 *
 * @author   Simply.IN Sp. z o.o.
 * @copyright 2024-2025 Simply.IN Sp. z o.o.
 * @license   https://joinup.ec.europa.eu/software/page/eupl
 */

class SimplyInApi
{
    private static $instance = null;
    public $api_log_request = false;
    private $simplyin_session = false;
    private $context = null;
    private $backend_url = false;
    private $merchand_api_key = false;
    private $version = false;


    public static function getInstance()
    {
        if (self::$instance === null) {
            self::$instance = new SimplyInApi();
            self::$instance->context = Context::getContext();
            self::$instance->simplyin_session = SimplyInSession::getInstance();
        }
        return self::$instance;
    }

    public function logRequests()
    {
        return $this->api_log_request;
    }

    public function setMerchantApiKey($merchand_api_key)
    {
		$this->merchand_api_key = $merchand_api_key;
    }

    public function getBackendUrl()
    {
        $module = Module::getInstanceByName('simplyin');
        self::$instance->version = $module->version;
        self::$instance->backend_url = $module->backend_url;
    }

    public function apiCall($body, $method, $endpoint, $token = null)
    {
        if ($this->backend_url === false) {
            $this->getBackendUrl();
        }
        $api_key = Configuration::get('SIMPLYIN_SECRET_KEY');
        if (!empty($body)) {
            $body['plugin_version'] = $this->version;
            $body['platform'] = 'PrestaShop';
            $body['shopName'] = Configuration::get('PS_SHOP_NAME');
            $body['shopVersion'] = Configuration::get('PS_VERSION_DB');
            $body['lng'] = strtoupper($this->context->language->iso_code);
            $body['apiKey'] = $api_key;
            $body['merchantApiKey'] = $api_key;
            $body['pluginVersion'] = $this->version;
        }

        $origin = Context::getContext()->shop->getBaseURL(true);

        if (empty($api_key)) {
            return [
                'error' => 'Simplyin API key is empty',
                'code' => 400,
            ];
        }

        if (!empty($token)) {
            $url = $this->backend_url . $endpoint . '?api_token=' . urlencode($token);
        } else {
            $url = $this->backend_url . $endpoint;
        }

        $headers = [];
        $headers[] = 'Content-Type: application/json';
        $headers[] = 'Origin: ' . $origin;
        if ($token === null) {
            $headers[] = 'X-Auth-Merchant-Api-Key: ' . $this->merchand_api_key;
        }

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);

        switch ($method) {
            case 'GET':
                curl_setopt($ch, CURLOPT_HTTPGET, 1);
                break;
            case 'POST':
                curl_setopt($ch, CURLOPT_POST, 1);
                break;
            case 'PATCH':
                curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PATCH');
                break;
            default:
                break;
        }
        if (!empty($body)) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($body));
        }
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $response = curl_exec($ch);
        $response = json_decode($response, true);
        curl_close($ch);
        $this->logApiRequest($method, $endpoint, $body, $response);
        return $response;
    }

	public function sendRequest(
	    $endpoint,
	    $method = 'GET',
	    $body_data = [],
	    $params = [],
	    $merchantApiKey = '',
	    $skipMerchantApiKey = false,
	    $auth_token = ''
	) {
        if ($this->backend_url === false) {
            $this->getBackendUrl();
        }
	    $apiKey = $merchantApiKey !== '' ? $merchantApiKey : (isset($this->merchand_api_key) && $this->merchand_api_key ? $this->merchand_api_key : Configuration::get('SIMPLYIN_SECRET_KEY'));
	    $this->merchand_api_key = $apiKey;

	    if (empty($this->merchand_api_key) && !$skipMerchantApiKey) {
	        http_response_code(400);
	        $o = new \stdClass();
	        $o->status = 'failed';
	        $o->message = 'Error: Simplyin API Key is empty';
	        $o->http_response_code = 400;
	        return $o;
	    }

	    $origin = Context::getContext()->shop->getBaseURL(true);
	    $headers = [
	        'Content-Type: application/json',
	        'Origin: ' . $origin,
	    ];

	    $methodU = strtoupper($method);
	    if (!$skipMerchantApiKey) {
	        $headers[] = 'X-Auth-Merchant-Api-Key: ' . $this->merchand_api_key;
	        if (in_array($methodU, ['POST', 'PUT', 'PATCH'])) {
	            if (!is_array($body_data)) { $body_data = []; }
	            $body_data['merchantApiKey'] = $this->merchand_api_key;
	        }
	    }

	    $baseUrl = '';

        $baseUrl = rtrim($this->backend_url, '/');

	    if ($baseUrl === '') {
	        $o = new \stdClass();
	        $o->status = 'failed';
	        $o->message = 'Error: backend_url is not configured';
	        $o->http_response_code = 500;
	        return $o;
	    }

	    $url = $baseUrl . '/' . ltrim($endpoint, '/');
	    if ($auth_token !== '') {
	        $url .= (strpos($url, '?') === false ? '?' : '&') . 'api_token=' . urlencode($auth_token);
	    }
	    if (!empty($params)) {
	        $url .= (strpos($url, '?') === false ? '?' : '&') . http_build_query($params);
	    }

	    $ch = curl_init();
	    if (in_array($methodU, ['POST', 'PUT', 'PATCH', 'DELETE'])) {
	        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $methodU);
	    }
	    if (in_array($methodU, ['POST', 'PUT', 'PATCH'])) {
	        $jsonData = json_encode($body_data);
	        curl_setopt($ch, CURLOPT_POSTFIELDS, $jsonData);
	    }

	    curl_setopt($ch, CURLOPT_URL, $url);
	    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
	    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
	    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
	    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
	    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
	    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);

	    $raw = curl_exec($ch);
	    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
	    curl_close($ch);


        $decoded = json_decode($raw);

	    if (method_exists($this, 'logApiRequest')) {
	        $this->logApiRequest($endpoint, $methodU, in_array($methodU, ['POST','PUT','PATCH']) ? (isset($jsonData) ? $jsonData : json_encode($body_data)) : null, $params, $decoded);
	    }

	    return $decoded;
	}


	public function logApiRequest($method, $endpoint, $request, $response)
    {
        if ($this->api_log_request) {
            $caller_file = debug_backtrace()[1]['file'];
            $caller_line = debug_backtrace()[1]['line'];

            $request = $this->convertBoolsToInt($request);
            $response = $this->convertBoolsToInt($response);

            $log = date('Y-m-d H:i:s');
            $log .= ' Caller: ' . $caller_file . ' (line ' . $caller_line . ")\n";
            $log .= 'Method: ' . $method . "\n";
            $log .= 'Endpoint: ' . $endpoint . "\n";
            $log .= 'Request: ' . print_r($request, true) . "\n";
            $log .= 'Response: ' . print_r($response, true) . "\n";
            $log .= "-------------------------\n";

            file_put_contents(dirname(__FILE__).'/../debug/api.log', $log, FILE_APPEND);
        }
    }

    public function getSecretKey($order_email)
    {
        return hash('sha256', '__' . $order_email . '__', true);
    }

    public function sendEncryptedData( $order_email, $order_reference, $new_order_state, $tracking_number)
    {
    	$apiKey = Configuration::get('SIMPLYIN_SECRET_KEY');
    	$body_data = [
            'email' => $order_email,
            'shopOrderNumber' => $order_reference,
            'newOrderStatus' => $new_order_state,
            'apiKey' => $apiKey,
            'trackings' => [$tracking_number],
        ];

        $plaintext = json_encode($body_data, JSON_UNESCAPED_SLASHES);

        $key = $this->getSecretKey($order_email);

        $encryptedData = $this->encrypt($plaintext, $key);

        $hashedEmail = $this->getEmailHash($order_email);

        $orderData = [];
        $orderData['encryptedOrderStatusChangeContent'] = $encryptedData;
        $orderData['hashedEmail'] = $hashedEmail;

        $url = $this->backend_url . 'encryption/saveEncryptedOrderStatusChange';
        $base_url = __PS_BASE_URI__;
        $headers = ['Content-Type: application/json', 'Origin: ' . $base_url];

        $ch = curl_init();

        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $orderData);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $response = curl_exec($ch);
        curl_close($ch);

        return $response;

    }

    public function encrypt($plaintext, $secret_key, $cipher = 'aes-256-cbc')
    {
        $ivlen = openssl_cipher_iv_length($cipher);
        $iv = openssl_random_pseudo_bytes($ivlen);

        $ciphertext_raw = openssl_encrypt($plaintext, $cipher, $secret_key, OPENSSL_RAW_DATA, $iv);
        if ($ciphertext_raw === false) {
            return false;
        }

        return base64_encode($iv . $ciphertext_raw);
    }

    public function getEmailHash($order_email)
    {
        return hash('sha256', '--' . $order_email . '--');
    }

    private function convertBoolsToInt($data) {
        if (is_array($data)) {
            foreach ($data as $key => $value) {
                $data[$key] = $this->convertBoolsToInt($value);
            }
        } elseif (is_object($data)) {
            foreach ($data as $key => $value) {
                $data->$key = $this->convertBoolsToInt($value);
            }
        } elseif (is_bool($data)) {
            return $data ? 1 : 0;
        }
        return $data;
    }

    public function getPaymentMethods()
    {
        $simplyin_auth_token = $this->simplyin_session->get('simplyin_auth_token');
        $auth_token = null;
        if (!empty($simplyin_auth_token)) {
            $auth_token = $simplyin_auth_token;
        }
        $result =  $this->sendRequest(
            'payments/getAvailablePaymentMethods',
            'GET',
            '',
            [],
            '',
            false,
            $auth_token
        );
        return $result;
    }

    public function simplyInChangeOrderState($id_order, $status)
    {
        $object = SimplyInOrder::getByIdOrder($id_order);
        if ($object) {
            if ($status == 'shipped') {
                $status = 'InTransit';
            } elseif ($status == 'payment') {
                $status = 'Pending';
            } else {
                return;
            }

            $body_data = [
                'orderId' => $object->simplyin_id,
                'status' => $status,
            ];

            return $this->sendRequest('orders/updateStatus', 'PUT', $body_data);
        }
    }
}